<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

try {
    $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
    $stmt->execute([$admin_id]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin) {
        header('Location: admin_login.php');
        exit;
    }
} catch (PDOException $e) {
    error_log("Admin Activation Codes error: " . $e->getMessage());
    header('Location: admin_dashboard.php');
    exit;
}

// DEBUG: Check current table structure
try {
    $debug_stmt = $pdo->query("DESCRIBE activation_codes");
    $table_structure = $debug_stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("Activation Codes Table Structure: " . print_r($table_structure, true));
} catch (Exception $e) {
    error_log("Error checking table structure: " . $e->getMessage());
}

// Handle form actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'generate_codes') {
        $quantity = intval($_POST['quantity'] ?? 1);
        $value = floatval($_POST['value'] ?? 0);
        $card_type = $_POST['card_type'] ?? 'premium';
        $expires_at = $_POST['expires_at'] ?? null;
        
        if ($quantity > 0 && $value > 0) {
            try {
                $pdo->beginTransaction();
                
                $generated_codes = [];
                for ($i = 0; $i < $quantity; $i++) {
                    // Generate unique code (more readable format)
                    $code = strtoupper('OX' . substr(md5(uniqid(mt_rand(), true)), 0, 6) . rand(1000, 9999));
                    
                    // Ensure uniqueness
                    $stmt = $pdo->prepare("SELECT id FROM activation_codes WHERE code = ?");
                    $stmt->execute([$code]);
                    if ($stmt->fetch()) {
                        $i--; // Try again if code exists
                        continue;
                    }
                    
                    // DEBUG: Check what columns exist
                    $check_columns = $pdo->query("SHOW COLUMNS FROM activation_codes")->fetchAll(PDO::FETCH_COLUMN);
                    error_log("Available columns: " . print_r($check_columns, true));
                    
                    // Try to insert with basic columns first
                    $sql = "INSERT INTO activation_codes (code";
                    $values = "VALUES (?";
                    $params = [$code];
                    
                    if (in_array('value', $check_columns)) {
                        $sql .= ", value";
                        $values .= ", ?";
                        $params[] = $value;
                    }
                    
                    if (in_array('card_type', $check_columns)) {
                        $sql .= ", card_type";
                        $values .= ", ?";
                        $params[] = $card_type;
                    }
                    
                    if (in_array('created_by', $check_columns)) {
                        $sql .= ", created_by";
                        $values .= ", ?";
                        $params[] = $admin_id;
                    }
                    
                    if (in_array('expires_at', $check_columns) && $expires_at) {
                        $sql .= ", expires_at";
                        $values .= ", ?";
                        $params[] = $expires_at;
                    }
                    
                    $sql .= ") " . $values . ")";
                    
                    error_log("Insert SQL: " . $sql);
                    error_log("Insert Params: " . print_r($params, true));
                    
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    
                    $generated_codes[] = $code;
                }
                
                $pdo->commit();
                $_SESSION['admin_success'] = "Successfully generated $quantity activation codes! Codes: " . implode(', ', $generated_codes);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $_SESSION['admin_error'] = "Error generating codes: " . $e->getMessage();
                error_log("Generation Error: " . $e->getMessage());
            }
        } else {
            $_SESSION['admin_error'] = "Please enter valid quantity and value!";
        }
    }
    
    elseif ($action === 'delete_code') {
        $code_id = $_POST['code_id'] ?? '';
        
        if (!empty($code_id)) {
            try {
                $stmt = $pdo->prepare("DELETE FROM activation_codes WHERE id = ?");
                $stmt->execute([$code_id]);
                
                if ($stmt->rowCount() > 0) {
                    $_SESSION['admin_success'] = "Activation code deleted successfully!";
                } else {
                    $_SESSION['admin_error'] = "Error deleting activation code!";
                }
                
            } catch (Exception $e) {
                $_SESSION['admin_error'] = "Error deleting code: " . $e->getMessage();
            }
        }
    }
    
    // Redirect to avoid form resubmission
    header('Location: admin_activation_codes.php');
    exit;
}

// Get activation codes - SIMPLIFIED VERSION
try {
    // First, let's just get all codes without complex joins
    $stmt = $pdo->query("SELECT * FROM activation_codes ORDER BY created_at DESC LIMIT 50");
    $activation_codes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("Found " . count($activation_codes) . " activation codes");
    if (!empty($activation_codes)) {
        error_log("Sample code: " . print_r($activation_codes[0], true));
    }
    
} catch (Exception $e) {
    error_log("Error fetching codes: " . $e->getMessage());
    $activation_codes = [];
}

$total_codes = count($activation_codes);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Activation Codes - Admin Dashboard</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-used {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-unused {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-expired {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .generate-form-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 800px;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Code copy animation */
        .code-copy {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .code-copy:hover {
            background-color: #e5e7eb !important;
        }
        
        .code-copy.copied {
            background-color: #d1fae5 !important;
            color: #065f46;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($admin['full_name'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($admin['full_name'] ?? 'Admin'); ?></span>
                        <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full hidden sm:inline"><?php echo ucfirst(str_replace('_', ' ', $admin['role'] ?? 'admin')); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-key text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Activation Codes</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Header Section -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Activation Codes Management 🔑</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Generate and manage premium virtual card activation codes</p>
                
                <!-- DEBUG INFO -->
                <div class="mt-2 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <p class="text-sm text-yellow-800"><strong>Debug:</strong> Found <?php echo $total_codes; ?> codes in database</p>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Generate Codes Form -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 border border-gray-200 animate-slide-up card-padding">
                <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 flex items-center">
                    <i class="fas fa-plus-circle text-green-600 mr-2"></i>
                    Generate New Activation Codes
                </h2>
                
                <form method="POST" class="grid grid-cols-1 md:grid-cols-4 gap-4 generate-form-grid">
                    <input type="hidden" name="action" value="generate_codes">
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Quantity</label>
                        <input type="number" name="quantity" min="1" max="100" value="5" 
                               class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent text-sm sm:text-base" required>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Value ($)</label>
                        <input type="number" name="value" min="0.01" step="0.01" value="50.00" 
                               class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent text-sm sm:text-base" required>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Card Type</label>
                        <select name="card_type" class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent text-sm sm:text-base" required>
                            <option value="standard">Standard</option>
                            <option value="premium" selected>Premium</option>
                            <option value="vip">VIP</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Expiry Date (Optional)</label>
                        <input type="date" name="expires_at" 
                               class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent text-sm sm:text-base">
                    </div>
                    
                    <div class="md:col-span-4">
                        <button type="submit" 
                                class="bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-6 rounded-lg transition-colors flex items-center justify-center text-sm sm:text-base shadow-sm">
                            <i class="fas fa-key mr-2"></i>
                            Generate Codes
                        </button>
                    </div>
                </form>
            </div>

            <!-- Quick Stats -->
            <?php if (!empty($activation_codes)): ?>
                <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 sm:gap-6 mb-6 stats-grid">
                    <div class="bg-gradient-to-r from-blue-50 to-blue-100 rounded-xl p-4 border border-blue-200 animate-slide-up">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-blue-600">Total Codes</p>
                                <p class="text-xl sm:text-2xl font-bold text-blue-900 mt-1"><?php echo $total_codes; ?></p>
                            </div>
                            <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                                <i class="fas fa-key text-blue-600 text-lg sm:text-xl"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="bg-gradient-to-r from-green-50 to-green-100 rounded-xl p-4 border border-green-200 animate-slide-up">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-green-600">Active Codes</p>
                                <p class="text-xl sm:text-2xl font-bold text-green-900 mt-1">
                                    <?php 
                                    $active_codes = array_filter($activation_codes, function($code) {
                                        return (!isset($code['is_used']) || !$code['is_used']) && 
                                               (!isset($code['expires_at']) || !$code['expires_at'] || strtotime($code['expires_at']) > time());
                                    });
                                    echo count($active_codes);
                                    ?>
                                </p>
                            </div>
                            <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                                <i class="fas fa-check-circle text-green-600 text-lg sm:text-xl"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="bg-gradient-to-r from-purple-50 to-purple-100 rounded-xl p-4 border border-purple-200 animate-slide-up">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-purple-600">Total Value</p>
                                <p class="text-xl sm:text-2xl font-bold text-purple-900 mt-1">
                                    $<?php 
                                    $total_value = array_sum(array_column($activation_codes, 'value'));
                                    echo number_format($total_value, 2);
                                    ?>
                                </p>
                            </div>
                            <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-purple-100 flex items-center justify-center">
                                <i class="fas fa-dollar-sign text-purple-600 text-lg sm:text-xl"></i>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Activation Codes Table -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden border border-gray-200 animate-slide-up">
                <div class="px-4 sm:px-6 py-4 border-b border-gray-200">
                    <h2 class="text-lg sm:text-xl font-bold text-gray-900 flex items-center">
                        <i class="fas fa-list mr-2 text-green-600"></i>
                        Activation Codes (<?php echo $total_codes; ?>)
                    </h2>
                </div>
                
                <?php if (!empty($activation_codes)): ?>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Value</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Created</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($activation_codes as $code): ?>
                                    <tr class="hover:bg-gray-50 transition-colors">
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-xs sm:text-sm text-gray-500">
                                            <?php echo $code['id']; ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <code class="code-copy text-xs sm:text-sm font-mono font-bold text-gray-900 bg-gray-100 px-2 py-1 rounded cursor-pointer" 
                                                  onclick="copyToClipboard(this, '<?php echo htmlspecialchars($code['code']); ?>')">
                                                <?php echo htmlspecialchars($code['code']); ?>
                                            </code>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="text-xs sm:text-sm font-bold text-green-600">
                                                $<?php echo number_format($code['value'] ?? 0, 2); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="text-xs sm:text-sm font-medium text-gray-700 capitalize">
                                                <?php echo htmlspecialchars($code['card_type'] ?? 'premium'); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <?php
                                            $status_class = 'status-unused';
                                            $status_text = 'Unused';
                                            
                                            if (isset($code['is_used']) && $code['is_used']) {
                                                $status_class = 'status-used';
                                                $status_text = 'Used';
                                            } elseif (isset($code['expires_at']) && $code['expires_at'] && strtotime($code['expires_at']) < time()) {
                                                $status_class = 'status-expired';
                                                $status_text = 'Expired';
                                            }
                                            ?>
                                            <span class="status-badge <?php echo $status_class; ?>">
                                                <?php echo $status_text; ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-xs sm:text-sm text-gray-500">
                                            <div class="flex flex-col">
                                                <span><?php echo date('M j, Y', strtotime($code['created_at'] ?? $code['created_at'])); ?></span>
                                                <span class="text-xs text-gray-400"><?php echo date('g:i A', strtotime($code['created_at'] ?? $code['created_at'])); ?></span>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-xs sm:text-sm font-medium">
                                            <?php if (!isset($code['is_used']) || !$code['is_used']): ?>
                                                <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this activation code?');">
                                                    <input type="hidden" name="action" value="delete_code">
                                                    <input type="hidden" name="code_id" value="<?php echo $code['id']; ?>">
                                                    <button type="submit" class="text-red-600 hover:text-red-900 transition-colors flex items-center">
                                                        <i class="fas fa-trash mr-1"></i> 
                                                        <span class="hidden sm:inline">Delete</span>
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <span class="text-gray-400 text-xs sm:text-sm">Used</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                <?php else: ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="w-12 h-12 sm:w-16 sm:h-16 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-key text-gray-400 text-lg sm:text-xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium text-base sm:text-lg">No activation codes found</p>
                        <p class="text-sm text-gray-400 mt-1 max-w-md mx-auto">Generate your first activation code using the form above</p>
                        
                        <!-- DEBUG: Check if there are actually codes in the database -->
                        <?php
                        try {
                            $check_stmt = $pdo->query("SELECT COUNT(*) as count FROM activation_codes");
                            $count_result = $check_stmt->fetch(PDO::FETCH_ASSOC);
                            if ($count_result['count'] > 0) {
                                echo '<div class="mt-4 p-3 bg-red-50 border border-red-200 rounded-lg max-w-md mx-auto">';
                                echo '<p class="text-xs sm:text-sm text-red-800"><strong>Database Conflict:</strong> There are ' . $count_result['count'] . ' codes in the database but they are not being displayed.</p>';
                                echo '</div>';
                            }
                        } catch (Exception $e) {
                            echo '<div class="mt-4 p-3 bg-red-50 border border-red-200 rounded-lg max-w-md mx-auto">';
                            echo '<p class="text-xs sm:text-sm text-red-800"><strong>Database Error:</strong> ' . $e->getMessage() . '</p>';
                            echo '</div>';
                        }
                        ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <script>
        function copyToClipboard(element, code) {
            navigator.clipboard.writeText(code).then(() => {
                element.classList.add('copied');
                element.textContent = 'Copied!';
                
                setTimeout(() => {
                    element.classList.remove('copied');
                    element.textContent = code;
                }, 2000);
            }).catch(err => {
                console.error('Failed to copy: ', err);
                alert('Failed to copy code to clipboard');
            });
        }

        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });
    </script>

</body>
</html>