<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Only super admin can access backup
if ($_SESSION['admin_role'] !== 'super_admin') {
    header('Location: admin_dashboard.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Create backups directory if it doesn't exist
$backup_dir = __DIR__ . '/backups/';
if (!file_exists($backup_dir)) {
    if (!mkdir($backup_dir, 0755, true)) {
        $_SESSION['admin_error'] = "Failed to create backup directory. Please check folder permissions.";
    }
}

// Handle backup actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'create_backup') {
        $backup_type = $_POST['backup_type'] ?? 'database';
        
        try {
            if ($backup_type === 'database') {
                $result = createDatabaseBackup($pdo, $admin_id);
            } elseif ($backup_type === 'files') {
                $result = createFilesBackup($pdo, $admin_id);
            } elseif ($backup_type === 'full') {
                $result = createFullBackup($pdo, $admin_id);
            }
            
            if ($result['success']) {
                $_SESSION['admin_success'] = $result['message'];
            } else {
                $_SESSION['admin_error'] = $result['message'];
            }
            
        } catch (Exception $e) {
            $_SESSION['admin_error'] = "Backup failed: " . $e->getMessage();
        }
    }
    
    elseif ($action === 'delete_backup') {
        $backup_file = $_POST['backup_file'] ?? '';
        $backup_path = __DIR__ . '/backups/' . $backup_file;
        
        if (!empty($backup_file) && file_exists($backup_path)) {
            if (unlink($backup_path)) {
                $_SESSION['admin_success'] = "Backup file deleted successfully!";
            } else {
                $_SESSION['admin_error'] = "Failed to delete backup file!";
            }
        } else {
            $_SESSION['admin_error'] = "Backup file not found!";
        }
    }
    
    elseif ($action === 'download_backup') {
        $backup_file = $_POST['backup_file'] ?? '';
        $backup_path = __DIR__ . '/backups/' . $backup_file;
        
        if (!empty($backup_file) && file_exists($backup_path)) {
            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . basename($backup_path) . '"');
            header('Expires: 0');
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            header('Content-Length: ' . filesize($backup_path));
            readfile($backup_path);
            exit;
        } else {
            $_SESSION['admin_error'] = "Backup file not found!";
            header('Location: admin_backup.php');
            exit;
        }
    }
    
    header('Location: admin_backup.php');
    exit;
}

// Backup functions
function createDatabaseBackup($pdo, $admin_id) {
    $backup_dir = __DIR__ . '/backups/';
    if (!file_exists($backup_dir)) {
        if (!mkdir($backup_dir, 0755, true)) {
            return [
                'success' => false,
                'message' => "Failed to create backup directory. Please check folder permissions."
            ];
        }
    }
    
    $timestamp = date('Y-m-d_H-i-s');
    $backup_file = $backup_dir . 'database_backup_' . $timestamp . '.sql';
    
    // Try mysqldump first
    $useMysqldump = true;
    $mysqldump_success = false;
    
    // Get database configuration from config constants
    $db_host = DB_HOST;
    $db_name = DB_NAME;
    $db_user = DB_USER;
    $db_pass = DB_PASS;
    
    // Try mysqldump if available
    if ($useMysqldump) {
        $command = "mysqldump --host={$db_host} --user={$db_user} --password={$db_pass} {$db_name} > \"{$backup_file}\" 2>&1";
        system($command, $output);
        
        if (file_exists($backup_file) && filesize($backup_file) > 0) {
            $mysqldump_success = true;
        } else {
            // Clean up empty file
            if (file_exists($backup_file)) {
                unlink($backup_file);
            }
        }
    }
    
    // If mysqldump failed, use PHP backup method
    if (!$mysqldump_success) {
        $backup_content = "";
        $backup_content .= "-- MySQL database backup\n";
        $backup_content .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
        $backup_content .= "-- Database: " . $db_name . "\n\n";
        
        // Get all tables
        try {
            $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
            
            foreach ($tables as $table) {
                $backup_content .= "--\n-- Table structure for table `$table`\n--\n\n";
                $backup_content .= "DROP TABLE IF EXISTS `$table`;\n";
                
                $create_table = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(PDO::FETCH_ASSOC);
                $backup_content .= $create_table['Create Table'] . ";\n\n";
                
                // Get table data
                $rows = $pdo->query("SELECT * FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
                
                if (count($rows) > 0) {
                    $backup_content .= "--\n-- Dumping data for table `$table`\n--\n\n";
                    
                    foreach ($rows as $row) {
                        $columns = array_keys($row);
                        $values = array_map(function($value) use ($pdo) {
                            if ($value === null) return 'NULL';
                            return $pdo->quote($value);
                        }, array_values($row));
                        
                        $backup_content .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $values) . ");\n";
                    }
                    $backup_content .= "\n";
                }
            }
            
            if (file_put_contents($backup_file, $backup_content) === false) {
                return [
                    'success' => false,
                    'message' => "Failed to write database backup file. Check folder permissions."
                ];
            }
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Database backup failed: " . $e->getMessage()
            ];
        }
    }
    
    if (file_exists($backup_file) && filesize($backup_file) > 0) {
        // Log backup activity
        logBackupActivity($pdo, $admin_id, 'database', $backup_file);
        return [
            'success' => true,
            'message' => "Database backup created successfully! File: " . basename($backup_file),
            'file' => $backup_file
        ];
    } else {
        return [
            'success' => false,
            'message' => "Failed to create database backup. Please check server permissions and MySQL credentials."
        ];
    }
}

function createFilesBackup($pdo, $admin_id) {
    $backup_dir = __DIR__ . '/backups/';
    if (!file_exists($backup_dir)) {
        if (!mkdir($backup_dir, 0755, true)) {
            return [
                'success' => false,
                'message' => "Failed to create backup directory. Please check folder permissions."
            ];
        }
    }
    
    $timestamp = date('Y-m-d_H-i-s');
    $backup_file = $backup_dir . 'files_backup_' . $timestamp . '.zip';
    
    // Check if ZipArchive is available
    if (!class_exists('ZipArchive')) {
        return [
            'success' => false,
            'message' => "ZIP extension is not available on this server. Please enable zip extension."
        ];
    }
    
    // Files to include in backup
    $files_to_backup = [
        __DIR__ . '/config.php',
        __DIR__ . '/admin_dashboard.php',
        __DIR__ . '/admin_users.php',
        __DIR__ . '/admin_transactions.php',
        __DIR__ . '/admin_support.php',
        __DIR__ . '/admin_funds.php',
        __DIR__ . '/admin_activation_codes.php',
        __DIR__ . '/admin_settings.php',
        __DIR__ . '/admin_backup.php',
        __DIR__ . '/admin_login.php',
        __DIR__ . '/admin_logout.php',
    ];
    
    $zip = new ZipArchive();
    if ($zip->open($backup_file, ZipArchive::CREATE) !== TRUE) {
        return [
            'success' => false,
            'message' => "Cannot create ZIP file. Check folder permissions."
        ];
    }
    
    $added_files = 0;
    
    foreach ($files_to_backup as $file) {
        if (file_exists($file)) {
            $relative_path = str_replace(__DIR__ . '/', '', $file);
            if ($zip->addFile($file, $relative_path)) {
                $added_files++;
            }
        }
    }
    
    // Add directories
    $directories_to_backup = [
        __DIR__ . '/includes/',
        __DIR__ . '/assets/',
        __DIR__ . '/uploads/'
    ];
    
    foreach ($directories_to_backup as $directory) {
        if (file_exists($directory)) {
            addDirectoryToZip($zip, $directory, basename($directory));
            $added_files++;
        }
    }
    
    $zip->close();
    
    if (file_exists($backup_file) && filesize($backup_file) > 0) {
        // Log backup activity
        logBackupActivity($pdo, $admin_id, 'files', $backup_file);
        return [
            'success' => true,
            'message' => "Files backup created successfully! File: " . basename($backup_file),
            'file' => $backup_file
        ];
    } else {
        // Clean up empty file
        if (file_exists($backup_file)) {
            unlink($backup_file);
        }
        return [
            'success' => false,
            'message' => "Failed to create files backup. No files were added to the backup."
        ];
    }
}

function createFullBackup($pdo, $admin_id) {
    $db_result = createDatabaseBackup($pdo, $admin_id);
    $files_result = createFilesBackup($pdo, $admin_id);
    
    if ($db_result['success'] && $files_result['success']) {
        return [
            'success' => true,
            'message' => "Full backup completed! Database: " . basename($db_result['file']) . ", Files: " . basename($files_result['file'])
        ];
    } else {
        $message = "Full backup completed with issues. ";
        if ($db_result['success']) {
            $message .= "Database: OK. ";
        } else {
            $message .= "Database failed: " . $db_result['message'] . " ";
        }
        if ($files_result['success']) {
            $message .= "Files: OK.";
        } else {
            $message .= "Files failed: " . $files_result['message'];
        }
        
        return [
            'success' => $db_result['success'] || $files_result['success'],
            'message' => $message
        ];
    }
}

function addDirectoryToZip($zip, $directory, $zip_dir = '') {
    if (!is_dir($directory)) return;
    
    $files = scandir($directory);
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        
        $file_path = $directory . '/' . $file;
        $zip_path = $zip_dir ? $zip_dir . '/' . $file : $file;
        
        if (is_dir($file_path)) {
            addDirectoryToZip($zip, $file_path, $zip_path);
        } else {
            $zip->addFile($file_path, $zip_path);
        }
    }
}

function logBackupActivity($pdo, $admin_id, $type, $file_path) {
    try {
        $stmt = $pdo->prepare("INSERT INTO backup_logs (admin_id, backup_type, file_path, file_size, created_at) VALUES (?, ?, ?, ?, NOW())");
        $file_size = file_exists($file_path) ? filesize($file_path) : 0;
        $stmt->execute([$admin_id, $type, basename($file_path), $file_size]);
        return true;
    } catch (Exception $e) {
        // If table doesn't exist, just continue without logging
        error_log("Failed to log backup activity: " . $e->getMessage());
        return false;
    }
}

// Get existing backups
function getBackupFiles() {
    $backup_dir = __DIR__ . '/backups/';
    if (!file_exists($backup_dir)) {
        return [];
    }
    
    $backups = [];
    $files = scandir($backup_dir);
    
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        
        $file_path = $backup_dir . $file;
        if (is_file($file_path)) {
            $backups[] = [
                'name' => $file,
                'path' => $file_path,
                'size' => filesize($file_path),
                'modified' => filemtime($file_path),
                'type' => pathinfo($file, PATHINFO_EXTENSION)
            ];
        }
    }
    
    // Sort by modification time (newest first)
    usort($backups, function($a, $b) {
        return $b['modified'] - $a['modified'];
    });
    
    return $backups;
}

// Get backup logs
function getBackupLogs($pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT bl.*, au.username as admin_username 
            FROM backup_logs bl 
            LEFT JOIN admin_users au ON bl.admin_id = au.id 
            ORDER BY bl.created_at DESC 
            LIMIT 50
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        // If table doesn't exist, return empty array
        return [];
    }
}

$backups = getBackupFiles();
$backup_logs = getBackupLogs($pdo);
$total_backups = count($backups);
$total_size = array_sum(array_column($backups, 'size'));

// Helper function to format file sizes
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Backup System - Oxland Credits</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .backup-grid {
                grid-template-columns: 1fr !important;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 800px;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Backup specific styles */
        .backup-card {
            transition: all 0.3s ease;
        }
        
        .backup-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        .file-type-database {
            border-left: 4px solid #3b82f6;
        }
        
        .file-type-files {
            border-left: 4px solid #10b981;
        }
        
        .file-type-zip {
            border-left: 4px solid #f59e0b;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">OXLAND</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                        <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full hidden sm:inline"><?php echo ucfirst($_SESSION['admin_role'] ?? 'admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_backup.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-database text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Backup System</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Super Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Backup System 💾</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Manage database and file backups for your system</p>
                <div class="mt-2 flex items-center text-xs sm:text-sm text-green-600">
                    <i class="fas fa-shield-alt mr-1"></i>
                    <span>Super Admin Access Only</span>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Backup Statistics -->
            <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 sm:gap-6 mb-6 sm:mb-8 stats-grid">
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-blue-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Backups</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $total_backups; ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-database text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-green-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Size</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1">
                                <?php echo formatBytes($total_size); ?>
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-hdd text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-purple-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Last Backup</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1">
                                <?php echo $total_backups > 0 ? date('M j', $backups[0]['modified']) : 'Never'; ?>
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-purple-100 flex items-center justify-center">
                            <i class="fas fa-clock text-purple-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Create Backup Section -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 animate-slide-up card-padding">
                <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 flex items-center">
                    <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                    Create New Backup
                </h2>
                
                <form method="POST" action="" class="space-y-4">
                    <input type="hidden" name="action" value="create_backup">
                    
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 backup-grid">
                        <!-- Database Backup -->
                        <div class="backup-card bg-gradient-to-r from-blue-50 to-blue-100 border border-blue-200 rounded-xl p-4 sm:p-6 text-center">
                            <div class="w-12 h-12 mx-auto mb-3 rounded-full bg-blue-100 flex items-center justify-center">
                                <i class="fas fa-database text-blue-600 text-xl"></i>
                            </div>
                            <h3 class="font-semibold text-gray-900 mb-2">Database Backup</h3>
                            <p class="text-sm text-gray-600 mb-4">Backup your MySQL database only</p>
                            <button type="submit" name="backup_type" value="database" 
                                    class="w-full bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 transition-colors font-medium text-sm">
                                <i class="fas fa-download mr-2"></i>
                                Backup Database
                            </button>
                        </div>
                        
                        <!-- Files Backup -->
                        <div class="backup-card bg-gradient-to-r from-green-50 to-green-100 border border-green-200 rounded-xl p-4 sm:p-6 text-center">
                            <div class="w-12 h-12 mx-auto mb-3 rounded-full bg-green-100 flex items-center justify-center">
                                <i class="fas fa-file-archive text-green-600 text-xl"></i>
                            </div>
                            <h3 class="font-semibold text-gray-900 mb-2">Files Backup</h3>
                            <p class="text-sm text-gray-600 mb-4">Backup source code and important files</p>
                            <button type="submit" name="backup_type" value="files" 
                                    class="w-full bg-green-600 text-white py-2 px-4 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm">
                                <i class="fas fa-file-zipper mr-2"></i>
                                Backup Files
                            </button>
                        </div>
                        
                        <!-- Full Backup -->
                        <div class="backup-card bg-gradient-to-r from-purple-50 to-purple-100 border border-purple-200 rounded-xl p-4 sm:p-6 text-center">
                            <div class="w-12 h-12 mx-auto mb-3 rounded-full bg-purple-100 flex items-center justify-center">
                                <i class="fas fa-server text-purple-600 text-xl"></i>
                            </div>
                            <h3 class="font-semibold text-gray-900 mb-2">Full Backup</h3>
                            <p class="text-sm text-gray-600 mb-4">Complete backup (Database + Files)</p>
                            <button type="submit" name="backup_type" value="full" 
                                    class="w-full bg-purple-600 text-white py-2 px-4 rounded-lg hover:bg-purple-700 transition-colors font-medium text-sm">
                                <i class="fas fa-layer-group mr-2"></i>
                                Full Backup
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Existing Backups -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-slide-up">
                <div class="px-4 sm:px-6 py-4 border-b border-gray-200">
                    <h2 class="text-lg sm:text-xl font-bold text-gray-900 flex items-center">
                        <i class="fas fa-history mr-2 text-green-600"></i>
                        Existing Backups (<?php echo $total_backups; ?>)
                    </h2>
                </div>
                
                <?php if (!empty($backups)): ?>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">File Name</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Size</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Modified</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($backups as $backup): ?>
                                    <?php
                                    $file_type = 'zip';
                                    if (strpos($backup['name'], 'database') !== false) {
                                        $file_type = 'database';
                                    } elseif (strpos($backup['name'], 'files') !== false) {
                                        $file_type = 'files';
                                    }
                                    ?>
                                    <tr class="hover:bg-gray-50 transition-colors file-type-<?php echo $file_type; ?>">
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <i class="fas fa-file-<?php echo $file_type === 'database' ? 'code' : ($file_type === 'files' ? 'archive' : 'zipper'); ?> text-<?php echo $file_type === 'database' ? 'blue' : ($file_type === 'files' ? 'green' : 'yellow'); ?>-600 mr-2"></i>
                                                <div class="text-sm font-medium text-gray-900 truncate max-w-[200px]">
                                                    <?php echo htmlspecialchars($backup['name']); ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="text-xs font-medium text-gray-700 capitalize">
                                                <?php echo $file_type; ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <?php echo formatBytes($backup['size']); ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <?php echo date('M j, Y g:i A', $backup['modified']); ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm font-medium">
                                            <div class="flex items-center space-x-2">
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="download_backup">
                                                    <input type="hidden" name="backup_file" value="<?php echo htmlspecialchars($backup['name']); ?>">
                                                    <button type="submit" class="text-blue-600 hover:text-blue-900 transition-colors flex items-center">
                                                        <i class="fas fa-download mr-1"></i>
                                                        <span class="hidden sm:inline">Download</span>
                                                    </button>
                                                </form>
                                                <span class="text-gray-300">|</span>
                                                <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this backup?');">
                                                    <input type="hidden" name="action" value="delete_backup">
                                                    <input type="hidden" name="backup_file" value="<?php echo htmlspecialchars($backup['name']); ?>">
                                                    <button type="submit" class="text-red-600 hover:text-red-900 transition-colors flex items-center">
                                                        <i class="fas fa-trash mr-1"></i>
                                                        <span class="hidden sm:inline">Delete</span>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="w-16 h-16 sm:w-20 sm:h-20 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-database text-gray-400 text-xl sm:text-2xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium text-base sm:text-lg">No backup files found</p>
                        <p class="text-sm text-gray-400 mt-1 max-w-md mx-auto">Create your first backup using the options above</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });
    </script>

</body>
</html>