<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $user_id = $_POST['user_id'] ?? '';
    
    if ($action === 'toggle_status' && !empty($user_id)) {
        try {
            // Get current status
            $stmt = $pdo->prepare("SELECT is_active FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                $new_status = $user['is_active'] ? 0 : 1;
                $stmt = $pdo->prepare("UPDATE users SET is_active = ? WHERE id = ?");
                $stmt->execute([$new_status, $user_id]);
                
                // Create notification
                $status_text = $new_status ? 'activated' : 'suspended';
                $stmt = $pdo->prepare("INSERT INTO user_notifications (user_id, title, message, type, sent_by) 
                                      VALUES (?, 'Account Status', ?, 'warning', ?)");
                $message = "Your account has been {$status_text} by administrator.";
                $stmt->execute([$user_id, $message, $admin_id]);
                
                $_SESSION['admin_success'] = "User status updated successfully!";
            }
        } catch (Exception $e) {
            $_SESSION['admin_error'] = "Error updating user: " . $e->getMessage();
        }
    }
    
    elseif ($action === 'update_account_status' && !empty($user_id)) {
        $account_status = $_POST['account_status'] ?? '';
        if (!empty($account_status)) {
            try {
                $stmt = $pdo->prepare("UPDATE accounts SET status = ? WHERE user_id = ?");
                $stmt->execute([$account_status, $user_id]);
                
                $_SESSION['admin_success'] = "Account status updated successfully!";
            } catch (Exception $e) {
                $_SESSION['admin_error'] = "Error updating account status: " . $e->getMessage();
            }
        }
    }
    
    header('Location: admin_users.php');
    exit;
}

// Get all users with their account info
try {
    $search = $_GET['search'] ?? '';
    $status_filter = $_GET['status'] ?? '';
    
    $query = "SELECT u.*, a.balance, a.account_number, a.status as account_status 
              FROM users u 
              LEFT JOIN accounts a ON u.id = a.user_id 
              WHERE 1=1";
    
    $params = [];
    
    if (!empty($search)) {
        $query .= " AND (u.full_name LIKE ? OR u.email LIKE ? OR u.phone LIKE ? OR a.account_number LIKE ?)";
        $search_term = "%$search%";
        $params = array_merge($params, [$search_term, $search_term, $search_term, $search_term]);
    }
    
    if ($status_filter === 'active') {
        $query .= " AND u.is_active = 1";
    } elseif ($status_filter === 'inactive') {
        $query .= " AND u.is_active = 0";
    }
    
    $query .= " ORDER BY u.created_at DESC";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Admin Users error: " . $e->getMessage());
    $users = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Reuse styles from dashboard */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-active {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-inactive {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-suspended {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 800px;
            }
            
            .filter-container {
                flex-direction: column;
                align-items: stretch;
            }
            
            .filter-actions {
                flex-direction: row;
                justify-content: space-between;
                margin-top: 1rem;
            }
            
            .user-actions {
                flex-wrap: wrap;
                gap: 0.5rem;
            }
            
            .user-actions .separator {
                display: none;
            }
        }
        
        @media (max-width: 640px) {
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
            
            .filter-actions {
                flex-direction: column;
                gap: 0.75rem;
            }
            
            .filter-actions button {
                width: 100%;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-users text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">User Management 👥</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Manage all user accounts and their status</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Search and Filter -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 animate-slide-up card-padding">
                <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 filter-container">
                    <div class="flex-1">
                        <div class="relative max-w-md">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                            <input type="text" id="search" placeholder="Search users by name, email, phone, or account number..." 
                                   value="<?php echo htmlspecialchars($search); ?>"
                                   class="pl-10 w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                        </div>
                    </div>
                    <div class="flex items-center space-x-4 filter-actions">
                        <select id="statusFilter" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base w-full sm:w-auto">
                            <option value="">All Status</option>
                            <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                        <button onclick="applyFilters()" class="bg-green-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base w-full sm:w-auto">
                            Apply Filters
                        </button>
                    </div>
                </div>
            </div>

            <!-- Users Table -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-slide-up">
                <?php if (!empty($users)): ?>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Account Info</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Balance</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Joined</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($users as $user): ?>
                                    <tr class="hover:bg-gray-50 transition-colors">
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-2 sm:mr-3">
                                                    <span class="text-green-700 font-bold text-sm"><?php echo strtoupper(substr($user['full_name'], 0, 1)); ?></span>
                                                </div>
                                                <div class="min-w-0 flex-1">
                                                    <div class="text-sm font-medium text-gray-900 truncate"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                                    <div class="text-sm text-gray-500 truncate"><?php echo htmlspecialchars($user['email']); ?></div>
                                                    <div class="text-xs text-gray-400 truncate"><?php echo htmlspecialchars($user['phone'] ?? 'N/A'); ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="text-sm text-gray-900 font-mono"><?php echo htmlspecialchars($user['account_number'] ?? 'N/A'); ?></div>
                                            <div class="text-sm text-gray-500 capitalize"><?php echo htmlspecialchars($user['account_type'] ?? 'Standard'); ?> Account</div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="flex flex-col space-y-1">
                                                <span class="status-badge status-<?php echo $user['is_active'] ? 'active' : 'inactive'; ?> inline-block">
                                                    <?php echo $user['is_active'] ? 'Active' : 'Inactive'; ?>
                                                </span>
                                                <span class="status-badge status-<?php echo $user['account_status'] ?? 'active'; ?> inline-block">
                                                    <?php echo ucfirst($user['account_status'] ?? 'active'); ?>
                                                </span>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="text-sm font-bold text-gray-900">$<?php echo number_format($user['balance'] ?? 0, 2); ?></div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <?php echo date('M j, Y', strtotime($user['created_at'])); ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm font-medium">
                                            <div class="flex items-center space-x-1 sm:space-x-2 user-actions">
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <input type="hidden" name="action" value="toggle_status">
                                                    <button type="submit" class="text-<?php echo $user['is_active'] ? 'red' : 'green'; ?>-600 hover:text-<?php echo $user['is_active'] ? 'red' : 'green'; ?>-800 text-xs sm:text-sm font-medium px-2 py-1 rounded border border-<?php echo $user['is_active'] ? 'red' : 'green'; ?>-200 hover:bg-<?php echo $user['is_active'] ? 'red' : 'green'; ?>-50 transition-colors">
                                                        <?php echo $user['is_active'] ? 'Suspend' : 'Activate'; ?>
                                                    </button>
                                                </form>
                                                <span class="text-gray-300 separator">|</span>
                                                <a href="admin_funds.php?user_id=<?php echo $user['id']; ?>" class="text-blue-600 hover:text-blue-800 text-xs sm:text-sm font-medium px-2 py-1 rounded border border-blue-200 hover:bg-blue-50 transition-colors">
                                                    Funds
                                                </a>
                                                <span class="text-gray-300 separator">|</span>
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <input type="hidden" name="action" value="update_account_status">
                                                    <select name="account_status" onchange="this.form.submit()" class="text-xs border border-gray-300 rounded px-2 py-1 focus:ring-1 focus:ring-green-500 focus:border-green-500">
                                                        <option value="active" <?php echo ($user['account_status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>Active</option>
                                                        <option value="inactive" <?php echo ($user['account_status'] ?? 'active') === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                                        <option value="suspended" <?php echo ($user['account_status'] ?? 'active') === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                                                    </select>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="w-16 h-16 sm:w-20 sm:h-20 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-users text-gray-400 text-xl sm:text-2xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium text-lg sm:text-xl mb-2">No users found</p>
                        <p class="text-gray-400 text-sm sm:text-base max-w-md mx-auto">
                            <?php echo (!empty($search) || !empty($status_filter)) ? 
                                'Try adjusting your search filters' : 
                                'No users have been registered yet'; ?>
                        </p>
                        <?php if (!empty($search) || !empty($status_filter)): ?>
                            <button onclick="clearFilters()" class="mt-4 bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm">
                                Clear Filters
                            </button>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- User Count -->
            <?php if (!empty($users)): ?>
                <div class="mt-4 text-sm text-gray-500 animate-slide-up">
                    Showing <span class="font-semibold"><?php echo count($users); ?></span> user<?php echo count($users) !== 1 ? 's' : ''; ?>
                    <?php if (!empty($search)): ?>
                        matching "<span class="font-semibold"><?php echo htmlspecialchars($search); ?></span>"
                    <?php endif; ?>
                    <?php if (!empty($status_filter)): ?>
                        with status "<span class="font-semibold"><?php echo ucfirst($status_filter); ?></span>"
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });

        function applyFilters() {
            const search = document.getElementById('search').value;
            const status = document.getElementById('statusFilter').value;
            
            let url = 'admin_users.php?';
            const params = [];
            
            if (search) params.push(`search=${encodeURIComponent(search)}`);
            if (status) params.push(`status=${encodeURIComponent(status)}`);
            
            window.location.href = url + params.join('&');
        }
        
        function clearFilters() {
            window.location.href = 'admin_users.php';
        }

        // Auto-submit search on enter
        document.getElementById('search').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                applyFilters();
            }
        });
    </script>

</body>
</html>