<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

try {
    $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
    $stmt->execute([$admin_id]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin) {
        header('Location: admin_login.php');
        exit;
    }
    
    // Get dashboard statistics
    $stats = [];
    
    // Total users
    $stmt = $pdo->query("SELECT COUNT(*) as total_users FROM users");
    $stats['total_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_users'];
    
    // Active users
    $stmt = $pdo->query("SELECT COUNT(*) as active_users FROM users WHERE is_active = 1");
    $stats['active_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['active_users'];
    
    // Total transactions
    $stmt = $pdo->query("SELECT COUNT(*) as total_transactions FROM transactions");
    $stats['total_transactions'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_transactions'];
    
    // Total balance
    $stmt = $pdo->query("SELECT SUM(balance) as total_balance FROM accounts WHERE status = 'active'");
    $stats['total_balance'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_balance'] ?? 0;
    
    // Pending deposits
    $stmt = $pdo->query("SELECT COUNT(*) as pending_deposits FROM transactions WHERE type = 'credit' AND status = 'pending'");
    $stats['pending_deposits'] = $stmt->fetch(PDO::FETCH_ASSOC)['pending_deposits'];
    
    // Open support tickets
    $stmt = $pdo->query("SELECT COUNT(*) as open_tickets FROM support_tickets WHERE status IN ('open', 'pending')");
    $stats['open_tickets'] = $stmt->fetch(PDO::FETCH_ASSOC)['open_tickets'];
    
    // Recent transactions
    $stmt = $pdo->prepare("SELECT t.*, u.full_name, u.email FROM transactions t 
                          JOIN users u ON t.user_id = u.id 
                          ORDER BY t.created_at DESC LIMIT 10");
    $stmt->execute();
    $recent_transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Recent users
    $stmt = $pdo->prepare("SELECT u.*, a.balance, a.account_number FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          ORDER BY u.created_at DESC LIMIT 10");
    $stmt->execute();
    $recent_users = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Admin Dashboard error: " . $e->getMessage());
    $stats = [
        'total_users' => 0,
        'active_users' => 0,
        'total_transactions' => 0,
        'total_balance' => 0,
        'pending_deposits' => 0,
        'open_tickets' => 0
    ];
    $recent_transactions = [];
    $recent_users = [];
}

// Handle fund management
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'manage_funds') {
        $user_id = $_POST['user_id'] ?? '';
        $amount = $_POST['amount'] ?? '';
        $type = $_POST['transaction_type'] ?? '';
        $description = $_POST['description'] ?? '';
        
        if (!empty($user_id) && !empty($amount) && !empty($type)) {
            try {
                $pdo->beginTransaction();
                
                // Get user account
                $stmt = $pdo->prepare("SELECT * FROM accounts WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $account = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($account) {
                    if ($type === 'credit') {
                        // Add funds
                        $stmt = $pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE user_id = ?");
                        $stmt->execute([$amount, $user_id]);
                        
                        $transaction_description = "Admin credit: " . $description;
                    } else {
                        // Debit funds (check if sufficient balance)
                        if ($account['balance'] >= $amount) {
                            $stmt = $pdo->prepare("UPDATE accounts SET balance = balance - ? WHERE user_id = ?");
                            $stmt->execute([$amount, $user_id]);
                            
                            $transaction_description = "Admin debit: " . $description;
                        } else {
                            throw new Exception("Insufficient funds");
                        }
                    }
                    
                    // Log transaction
                    $stmt = $pdo->prepare("INSERT INTO transactions (user_id, type, amount, description, status) 
                                          VALUES (?, ?, ?, ?, 'completed')");
                    $stmt->execute([$user_id, $type, $amount, $transaction_description]);
                    
                    // Create user notification
                    $stmt = $pdo->prepare("INSERT INTO user_notifications (user_id, title, message, type, sent_by) 
                                          VALUES (?, 'Account Update', ?, 'info', ?)");
                    $notification_message = "Your account has been {$type}ed with amount: $" . number_format($amount, 2) . ". " . $description;
                    $stmt->execute([$user_id, $notification_message, $admin_id]);
                    
                    $pdo->commit();
                    $_SESSION['admin_success'] = "Funds {$type}ed successfully!";
                } else {
                    throw new Exception("User account not found");
                }
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $_SESSION['admin_error'] = "Error: " . $e->getMessage();
            }
        }
    }
    
    elseif ($action === 'update_account_status') {
        $user_id = $_POST['user_id'] ?? '';
        $status = $_POST['account_status'] ?? '';
        
        if (!empty($user_id) && !empty($status)) {
            try {
                $stmt = $pdo->prepare("UPDATE accounts SET status = ? WHERE user_id = ?");
                $stmt->execute([$status, $user_id]);
                
                // Get user email for notification
                $stmt = $pdo->prepare("SELECT email FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Create user notification
                $stmt = $pdo->prepare("INSERT INTO user_notifications (user_id, title, message, type, sent_by) 
                                      VALUES (?, 'Account Status Update', ?, 'warning', ?)");
                $notification_message = "Your account status has been updated to: " . ucfirst($status);
                $stmt->execute([$user_id, $notification_message, $admin_id]);
                
                $_SESSION['admin_success'] = "Account status updated successfully!";
                
            } catch (Exception $e) {
                $_SESSION['admin_error'] = "Error updating account status: " . $e->getMessage();
            }
        }
    }
    
    // Redirect to avoid form resubmission
    header('Location: admin_dashboard.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        /* Status badges */
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-active {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-inactive {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-suspended {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        .status-pending {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-completed {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        /* Transaction type indicators */
        .transaction-type-credit {
            border-left: 4px solid #10b981;
        }
        
        .transaction-type-debit {
            border-left: 4px solid #ef4444;
        }
        
        /* Tab styling */
        .tab-button {
            transition: all 0.3s ease;
            border-radius: 12px;
        }
        
        .tab-button.active {
            background-color: #22c55e;
            color: white;
        }
        
        /* Loading spinner */
        .spinner {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .main-grid {
                grid-template-columns: 1fr !important;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 640px;
            }
            
            .user-card-grid {
                grid-template-columns: 1fr !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($admin['full_name'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($admin['full_name'] ?? 'Admin'); ?></span>
                        <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full hidden sm:inline"><?php echo ucfirst(str_replace('_', ' ', $admin['role'] ?? 'admin')); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-tachometer-alt text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Welcome Section -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900 text-sharp">Admin Dashboard 🛠️</h1>
                <p class="text-gray-600 mt-1 font-medium text-sm sm:text-base">Welcome back, <?php echo htmlspecialchars($admin['full_name']); ?>! Here's your system overview.</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6 mb-8 stats-grid">
                <!-- Total Users -->
                <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Users</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['total_users']; ?></p>
                            <p class="text-xs text-green-600 font-medium mt-1">
                                <i class="fas fa-users mr-1"></i>
                                <?php echo $stats['active_users']; ?> active
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-users text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <!-- Total Balance -->
                <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Balance</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1">$<?php echo number_format($stats['total_balance'], 2); ?></p>
                            <p class="text-xs text-gray-500 font-medium mt-1">Across all accounts</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-wallet text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <!-- Transactions -->
                <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Transactions</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['total_transactions']; ?></p>
                            <p class="text-xs text-yellow-600 font-medium mt-1">
                                <i class="fas fa-clock mr-1"></i>
                                <?php echo $stats['pending_deposits']; ?> pending
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-purple-100 flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-purple-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <!-- Support Tickets -->
                <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Support Tickets</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['open_tickets']; ?></p>
                            <p class="text-xs text-red-600 font-medium mt-1">
                                <i class="fas fa-exclamation-circle mr-1"></i>
                                Needs attention
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-red-100 flex items-center justify-center">
                            <i class="fas fa-headset text-red-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions & Recent Activity -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 sm:gap-8 main-grid">
                
                <!-- Quick Actions -->
                <div class="lg:col-span-1">
                    <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up card-padding">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 sm:mb-5 flex items-center">
                            <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                            Quick Actions
                        </h2>
                        
                        <div class="space-y-3">
                            <a href="admin_funds.php" class="flex items-center p-3 sm:p-4 bg-white rounded-xl border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-lg bg-green-100 flex items-center justify-center mr-2 sm:mr-3">
                                    <i class="fas fa-money-bill-wave text-green-600 text-sm sm:text-base"></i>
                                </div>
                                <div>
                                    <p class="text-xs sm:text-sm font-semibold text-gray-900">Manage Funds</p>
                                    <p class="text-xs text-gray-500 font-medium">Add/remove user funds</p>
                                </div>
                            </a>
                            
                            <a href="admin_users.php" class="flex items-center p-3 sm:p-4 bg-white rounded-xl border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-lg bg-blue-100 flex items-center justify-center mr-2 sm:mr-3">
                                    <i class="fas fa-user-cog text-blue-600 text-sm sm:text-base"></i>
                                </div>
                                <div>
                                    <p class="text-xs sm:text-sm font-semibold text-gray-900">User Management</p>
                                    <p class="text-xs text-gray-500 font-medium">View and manage users</p>
                                </div>
                            </a>
                            
                            <a href="admin_support.php" class="flex items-center p-3 sm:p-4 bg-white rounded-xl border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-lg bg-purple-100 flex items-center justify-center mr-2 sm:mr-3">
                                    <i class="fas fa-ticket-alt text-purple-600 text-sm sm:text-base"></i>
                                </div>
                                <div>
                                    <p class="text-xs sm:text-sm font-semibold text-gray-900">Support Tickets</p>
                                    <p class="text-xs text-gray-500 font-medium">Handle customer support</p>
                                </div>
                            </a>
                            
                            <a href="admin_reports.php" class="flex items-center p-3 sm:p-4 bg-white rounded-xl border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-lg bg-yellow-100 flex items-center justify-center mr-2 sm:mr-3">
                                    <i class="fas fa-chart-line text-yellow-600 text-sm sm:text-base"></i>
                                </div>
                                <div>
                                    <p class="text-xs sm:text-sm font-semibold text-gray-900">Generate Reports</p>
                                    <p class="text-xs text-gray-500 font-medium">System analytics</p>
                                </div>
                            </a>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Transactions -->
                <div class="lg:col-span-2">
                    <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm animate-slide-up card-padding">
                        <div class="flex justify-between items-center mb-4 sm:mb-5">
                            <h2 class="text-lg sm:text-xl font-bold text-gray-900 flex items-center">
                                <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                                Recent Transactions
                            </h2>
                            <a href="admin_transactions.php" class="text-green-600 hover:text-green-800 font-medium text-xs sm:text-sm flex items-center">
                                View All
                                <i class="fas fa-arrow-right ml-1 text-xs"></i>
                            </a>
                        </div>
                        
                        <div class="overflow-hidden rounded-xl border border-gray-200">
                            <?php if (!empty($recent_transactions)): ?>
                                <div class="table-responsive">
                                    <table class="min-w-full divide-y divide-gray-200">
                                        <thead class="bg-gray-50">
                                            <tr>
                                                <th class="px-3 py-2 sm:px-4 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                                                <th class="px-3 py-2 sm:px-4 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                                                <th class="px-3 py-2 sm:px-4 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                                                <th class="px-3 py-2 sm:px-4 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                                <th class="px-3 py-2 sm:px-4 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                                            </tr>
                                        </thead>
                                        <tbody class="bg-white divide-y divide-gray-200">
                                            <?php foreach ($recent_transactions as $transaction): ?>
                                                <tr class="hover:bg-gray-50 transition-colors <?php echo $transaction['type'] === 'credit' ? 'transaction-type-credit' : 'transaction-type-debit'; ?>">
                                                    <td class="px-3 py-2 sm:px-4 sm:py-3 whitespace-nowrap">
                                                        <div class="text-xs sm:text-sm font-medium text-gray-900"><?php echo htmlspecialchars($transaction['full_name']); ?></div>
                                                        <div class="text-xs text-gray-500 truncate max-w-[120px] sm:max-w-none"><?php echo htmlspecialchars($transaction['email']); ?></div>
                                                    </td>
                                                    <td class="px-3 py-2 sm:px-4 sm:py-3 whitespace-nowrap">
                                                        <span class="text-xs sm:text-sm font-medium <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                            <?php echo ucfirst($transaction['type']); ?>
                                                        </span>
                                                    </td>
                                                    <td class="px-3 py-2 sm:px-4 sm:py-3 whitespace-nowrap">
                                                        <span class="text-xs sm:text-sm font-bold <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                            <?php echo $transaction['type'] === 'credit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                                        </span>
                                                    </td>
                                                    <td class="px-3 py-2 sm:px-4 sm:py-3 whitespace-nowrap">
                                                        <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                                            <?php echo ucfirst($transaction['status']); ?>
                                                        </span>
                                                    </td>
                                                    <td class="px-3 py-2 sm:px-4 sm:py-3 whitespace-nowrap text-xs sm:text-sm text-gray-500">
                                                        <?php echo date('M j, Y', strtotime($transaction['created_at'])); ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-6 sm:py-8">
                                    <div class="w-12 h-12 sm:w-16 sm:h-16 mx-auto mb-3 sm:mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-exchange-alt text-gray-400 text-lg sm:text-xl"></i>
                                    </div>
                                    <p class="text-gray-500 font-medium text-sm sm:text-base">No recent transactions</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Recent Users -->
                    <div class="gradient-card rounded-xl p-4 sm:p-6 shadow-sm mt-4 sm:mt-6 animate-slide-up card-padding">
                        <div class="flex justify-between items-center mb-4 sm:mb-5">
                            <h2 class="text-lg sm:text-xl font-bold text-gray-900 flex items-center">
                                <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                                Recent Users
                            </h2>
                            <a href="admin_users.php" class="text-green-600 hover:text-green-800 font-medium text-xs sm:text-sm flex items-center">
                                View All
                                <i class="fas fa-arrow-right ml-1 text-xs"></i>
                            </a>
                        </div>
                        
                        <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4 user-card-grid">
                            <?php if (!empty($recent_users)): ?>
                                <?php foreach ($recent_users as $user): ?>
                                    <div class="bg-white border border-gray-200 rounded-xl p-3 sm:p-4 hover:shadow-md transition-shadow">
                                        <div class="flex items-center justify-between mb-2">
                                            <div class="flex items-center">
                                                <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-2 sm:mr-3">
                                                    <span class="text-green-700 font-bold text-sm"><?php echo strtoupper(substr($user['full_name'], 0, 1)); ?></span>
                                                </div>
                                                <div>
                                                    <p class="text-xs sm:text-sm font-semibold text-gray-900 truncate max-w-[100px] sm:max-w-none"><?php echo htmlspecialchars($user['full_name']); ?></p>
                                                    <p class="text-xs text-gray-500 truncate max-w-[120px] sm:max-w-none"><?php echo htmlspecialchars($user['email']); ?></p>
                                                </div>
                                            </div>
                                            <span class="status-badge status-<?php echo $user['is_active'] ? 'active' : 'inactive'; ?>">
                                                <?php echo $user['is_active'] ? 'Active' : 'Inactive'; ?>
                                            </span>
                                        </div>
                                        <div class="flex justify-between items-center text-xs text-gray-500">
                                            <span class="truncate max-w-[100px]">Acc: <?php echo htmlspecialchars($user['account_number'] ?? 'N/A'); ?></span>
                                            <span>Balance: $<?php echo number_format($user['balance'] ?? 0, 2); ?></span>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="col-span-2 text-center py-6 sm:py-8">
                                    <div class="w-12 h-12 sm:w-16 sm:h-16 mx-auto mb-3 sm:mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-users text-gray-400 text-lg sm:text-xl"></i>
                                    </div>
                                    <p class="text-gray-500 font-medium text-sm sm:text-base">No users found</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Mobile menu functionality
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            mobileMenuBtn.addEventListener('click', toggleSidebar);
            overlay.addEventListener('click', toggleSidebar);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
        });
    </script>

</body>
</html>