<?php
require_once 'config.php';

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = "Security token invalid. Please try again.";
    }
    
    // Validate terms agreement
    if (!isset($_POST['terms'])) {
        $errors[] = "You must agree to the Terms and Conditions.";
    }
    
    // Sanitize inputs
    $full_name = sanitizeInput($_POST['full_name'] ?? '');
    $email = sanitizeInput($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $phone = sanitizeInput($_POST['phone'] ?? '');
    $account_type = sanitizeInput($_POST['account_type'] ?? 'personal');
    
    // Validate inputs
    if (empty($full_name)) {
        $errors[] = "Full name is required.";
    }
    
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Valid email address is required.";
    }
    
    if (empty($password)) {
        $errors[] = "Password is required.";
    } elseif (strlen($password) < 8) {
        $errors[] = "Password must be at least 8 characters long.";
    }
    
    if ($password !== $confirm_password) {
        $errors[] = "Passwords do not match.";
    }
    
    // If no errors, proceed with registration
    if (empty($errors)) {
        try {
            $pdo = getDBConnection();
            
            // Check if email already exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->rowCount() > 0) {
                $errors[] = "Email address is already registered.";
            } else {
                // Hash password
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                // Generate account number
                $account_number = 'OXL' . str_pad(mt_rand(1, 999999999), 9, '0', STR_PAD_LEFT);
                
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Insert user
                    $stmt = $pdo->prepare("INSERT INTO users (email, password, full_name, phone, account_type) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$email, $hashed_password, $full_name, $phone, $account_type]);
                    $user_id = $pdo->lastInsertId();
                    
                    // Create default account
                    $stmt = $pdo->prepare("INSERT INTO accounts (user_id, account_number, account_type, balance) VALUES (?, ?, ?, ?)");
                    $default_account_type = ($account_type === 'business') ? 'business' : 'checking';
                    $stmt->execute([$user_id, $account_number, $default_account_type, 0.00]);
                    
                    // Generate verification token
                    $verification_token = generateVerificationToken($user_id, 'email_verification');
                    
                    if ($verification_token) {
                        $verification_url = BASE_URL . "/verify-email.php?token=" . $verification_token;
                        
                        // Send verification email
                        $email_sent = sendVerificationEmail($email, $full_name, $verification_url);
                        
                        // 
                        // --- START OF FIX ---
                        //
                        // We MUST check if the email was sent. If not, we roll back
                        // the transaction and show an error.
                        //
                        if ($email_sent) {
                            // Try to update verification sent time
                            try {
                                $stmt = $pdo->prepare("UPDATE users SET verification_sent_at = NOW() WHERE id = ?");
                                $stmt->execute([$user_id]);
                            } catch (PDOException $e) {
                                // Ignore error if column doesn't exist
                                error_log("Note: verification_sent_at column may not exist: " . $e->getMessage());
                            }

                            // Commit transaction
                            $pdo->commit();
                            
                            $success = true;
                            
                            // Log the registration
                            error_log("New user registered successfully: $email (User ID: $user_id)");

                        } else {
                            // EMAIL FAILED: Roll back all database changes
                            $pdo->rollBack();
                            $errors[] = "Registration failed: We could not send a verification email at this time. Please try again later.";
                            error_log("CRITICAL: Email failed to send for new user: $email. Registration rolled back.");
                        }
                        //
                        // --- END OF FIX ---
                        //

                    } else {
                        // This should rarely happen
                        $pdo->rollBack();
                        $errors[] = "Could not generate a verification token. Please try again.";
                    }
                    
                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
            }
        } catch (Exception $e) {
            $errors[] = "Registration failed. Please try again.";
            error_log("Registration error: " . $e->getMessage());
            
            // Debug information
            if (DEBUG_MODE) {
                $errors[] = "Debug: " . $e->getMessage();
            }
        }
    }
}

// If registration successful, redirect to success page
if ($success) {
    $_SESSION['registration_email'] = $email;
    $_SESSION['registration_success'] = true;
    redirect('verification-sent.php');
    exit;
}

// Email template function
function sendVerificationEmail($to, $name, $verification_url) {
    $subject = "Verify Your Email - Airwaves Credits";
    
    $body = "
    <!DOCTYPE html>
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #0d3b5d; color: white; padding: 20px; text-align: center; }
            .content { background: #f9f9f9; padding: 20px; }
            .button { background: #0d3b5d; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
            .footer { text-align: center; margin-top: 20px; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>Airwaves Credit Union</h1>
            </div>
            <div class='content'>
                <h2>Verify Your Email Address</h2>
                <p>Hello $name,</p>
                <p>Thank you for registering with Airwaves Credit Union. To complete your registration, please verify your email address by clicking the button below:</p>
                <p style='text-align: center;'>
                    <a href='$verification_url' class='button'>Verify Email Address</a>
                </p>
                <p>Or copy and paste this link in your browser:<br>
                <code>$verification_url</code></p>
                <p>This verification link will expire in 24 hours.</p>
                <p>If you didn't create an account with Airwaves Credit Union, please ignore this email.</p>
            </div>
            <div class='footer'>
                <p>&copy; 2025 Airwaves Credit Union. All rights reserved.</p>
            </div>
        </div>
    </body>
    </html>";
    
    // This function is in your config.php
    return sendEmail($to, $subject, $body);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-blue: #0d3b5d;
            --secondary-blue: #1e4e75;
            --accent-blue: #2c5f8a;
        }
        
        body { 
            font-family: 'Inter', sans-serif; 
            background: linear-gradient(135deg, #0d3b5d 0%, #1e4e75 50%, #2c5f8a 100%);
            min-height: 100vh;
            position: relative;
            overflow-x: hidden;
        }
        
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1200 800" opacity="0.03"><path fill="%23ffffff" d="M0,0H1200V800H0Z"/><circle cx="200" cy="200" r="150" fill="%23ffffff"/><circle cx="800" cy="600" r="120" fill="%23ffffff"/><rect x="400" y="100" width="300" height="200" fill="%23ffffff"/></svg>');
            background-size: cover;
            background-position: center;
            pointer-events: none;
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .floating-element {
            animation: float 6s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        .senior-illustration {
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 200 200" opacity="0.1"><path fill="%23ffffff" d="M100,50c27.61,0,50,22.39,50,50s-22.39,50-50,50S50,127.61,50,100,72.39,50,100,50Zm0,10c-22.09,0-40,17.91-40,40s17.91,40,40,40,40-17.91,40-40S122.09,60,100,60Z"/><circle cx="85" cy="85" r="5" fill="%23ffffff"/><circle cx="115" cy="85" r="5" fill="%23ffffff"/><path fill="%23ffffff" d="M80,115c0-11.05,8.95-20,20-20s20,8.95,20,20Z"/></svg>');
            background-size: contain;
            background-repeat: no-repeat;
            background-position: center;
        }
        
        .logo-circle {
            width: 120px;
            height: 120px;
            background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1), 0 0 0 1px rgba(255, 255, 255, 0.2);
            border: 4px solid rgba(255, 255, 255, 0.3);
        }
        
        .logo-image {
            width: 70px;
            height: 70px;
            object-fit: contain;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.2));
        }
    </style>
</head>
<body class="flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <!-- Background Elements -->
    <div class="absolute inset-0 overflow-hidden">
        <div class="absolute -top-40 -right-32 w-80 h-80 bg-white rounded-full opacity-10 floating-element"></div>
        <div class="absolute top-1/2 -left-20 w-60 h-60 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -2s;"></div>
        <div class="absolute bottom-20 right-1/4 w-40 h-40 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -4s;"></div>
        
        <!-- Senior Illustration Elements -->
        <div class="absolute top-20 left-10 w-32 h-32 senior-illustration opacity-20 floating-element"></div>
        <div class="absolute bottom-32 right-20 w-24 h-24 senior-illustration opacity-20 floating-element" style="animation-delay: -3s;"></div>
        <div class="absolute top-1/3 right-1/4 w-28 h-28 senior-illustration opacity-20 floating-element" style="animation-delay: -1.5s;"></div>
    </div>

    <div class="max-w-md w-full space-y-8 relative z-10">
        <!-- Header -->
        <div class="text-center">
            <!-- Logo with Circular Background -->
            <div class="flex justify-center mb-6">
                <div class="logo-circle floating-element">
                    <img src="Uploads/images/logo.png" alt="Airwaves Credit Union" class="logo-image">
                </div>
            </div>
            <h2 class="text-center text-3xl font-extrabold text-white">
                Create Account
            </h2>
            <p class="mt-2 text-center text-lg text-blue-100">
                Join thousands who trust Airwaves Credit Union
            </p>
        </div>

        <?php if (!empty($errors)): ?>
            <div class="glass-effect rounded-2xl p-4 border-l-4 border-red-500">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-circle text-red-500 text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-gray-800">
                            Please fix the following errors:
                        </h3>
                        <div class="mt-2 text-sm text-gray-700">
                            <ul class="list-disc list-inside space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <form class="glass-effect rounded-3xl p-8 space-y-6" method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div class="space-y-5">
                <div>
                    <label for="full_name" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-user text-[#0d3b5d] mr-2"></i>
                        Full Name
                    </label>
                    <input id="full_name" name="full_name" type="text" required 
                           class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                           placeholder="Enter your full name"
                           value="<?php echo htmlspecialchars($_POST['full_name'] ?? ''); ?>">
                    <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <i class="fas fa-user text-gray-400"></i>
                    </div>
                </div>

                <div>
                    <label for="email" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-envelope text-[#0d3b5d] mr-2"></i>
                        Email Address
                    </label>
                    <div class="relative">
                        <input id="email" name="email" type="email" required 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="Enter your email"
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-envelope text-gray-400"></i>
                        </div>
                    </div>
                </div>

                <div>
                    <label for="phone" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-phone text-[#0d3b5d] mr-2"></i>
                        Phone Number (Optional)
                    </label>
                    <div class="relative">
                        <input id="phone" name="phone" type="tel" 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="+1 (555) 123-4567"
                               value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-phone text-gray-400"></i>
                        </div>
                    </div>
                </div>

                <div>
                    <label for="account_type" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-building text-[#0d3b5d] mr-2"></i>
                        Account Type
                    </label>
                    <div class="relative">
                        <select id="account_type" name="account_type" 
                                class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80 appearance-none">
                            <option value="personal" <?php echo ($_POST['account_type'] ?? 'personal') === 'personal' ? 'selected' : ''; ?>>Personal Banking</option>
                            <option value="business" <?php echo ($_POST['account_type'] ?? '') === 'business' ? 'selected' : ''; ?>>Business Banking</option>
                        </select>
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-building text-gray-400"></i>
                        </div>
                        <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                            <i class="fas fa-chevron-down text-gray-400"></i>
                        </div>
                    </div>
                </div>

                <div>
                    <label for="password" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-lock text-[#0d3b5d] mr-2"></i>
                        Password
                    </label>
                    <div class="relative">
                        <input id="password" name="password" type="password" required 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="Create a password (min. 8 characters)">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-key text-gray-400"></i>
                        </div>
                    </div>
                </div>

                <div>
                    <label for="confirm_password" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-lock text-[#0d3b5d] mr-2"></i>
                        Confirm Password
                    </label>
                    <div class="relative">
                        <input id="confirm_password" name="confirm_password" type="password" required 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="Confirm your password">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-key text-gray-400"></i>
                        </div>
                    </div>
                </div>
            </div>

            <div class="flex items-center">
                <input id="terms" name="terms" type="checkbox" required
                       class="h-4 w-4 text-[#0d3b5d] focus:ring-[#0d3b5d] border-gray-300 rounded">
                <label for="terms" class="ml-2 block text-sm text-gray-700 font-medium">
                    I agree to the <a href="#" class="text-[#0d3b5d] hover:text-blue-800 font-semibold">Terms and Conditions</a> and <a href="#" class="text-[#0d3b5d] hover:text-blue-800 font-semibold">Privacy Policy</a>
                </label>
            </div>

            <div>
                <button type="submit" 
                        class="group relative w-full flex justify-center py-4 px-4 border border-transparent text-lg font-semibold rounded-xl text-white bg-[#0d3b5d] hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#0d3b5d] transition-all duration-300 transform hover:scale-[1.02] shadow-lg">
                    <span class="absolute left-0 inset-y-0 flex items-center pl-4">
                        <i class="fas fa-user-plus text-blue-200 group-hover:text-white transition-colors duration-300"></i>
                    </span>
                    Create Account
                </button>
            </div>

            <div class="text-center pt-4">
                <p class="text-sm text-gray-600 font-medium">
                    Already have an account?
                    <a href="login.php" class="font-bold text-[#0d3b5d] hover:text-blue-800 transition-colors duration-300 ml-1">
                        Sign in here
                    </a>
                </p>
            </div>
        </form>

        <!-- Security Notice -->
        <div class="text-center">
            <div class="inline-flex items-center space-x-2 text-xs text-blue-200 bg-blue-900/30 px-4 py-2 rounded-full">
                <i class="fas fa-shield-alt"></i>
                <span>Bank-level security & 256-bit encryption</span>
            </div>
        </div>

        <!-- Senior-Friendly Features -->
        <div class="glass-effect rounded-2xl p-6 mt-6">
            <div class="flex items-center justify-between">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-[#0d3b5d] rounded-full flex items-center justify-center">
                        <i class="fas fa-heart text-white text-sm"></i>
                    </div>
                    <div>
                        <p class="text-sm font-semibold text-gray-800">Senior-Friendly</p>
                        <p class="text-xs text-gray-600">Easy registration for all generations</p>
                    </div>
                </div>
                <div class="text-2xl">👵👴</div>
            </div>
        </div>
    </div>

    <script>
        // Enhanced form interactions
        document.addEventListener('DOMContentLoaded', function() {
            const inputs = document.querySelectorAll('input, select');
            
            inputs.forEach(input => {
                input.addEventListener('focus', function() {
                    this.parentElement.classList.add('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
                
                input.addEventListener('blur', function() {
                    this.parentElement.classList.remove('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
            });
            
            // Auto-focus first input
            document.getElementById('full_name')?.focus();
            
            // Add subtle background animation
            const floatingElements = document.querySelectorAll('.floating-element');
            floatingElements.forEach((element, index) => {
                element.style.animationDelay = `${-index * 2}s`;
            });
        });
    </script>
</body>
</html>