<?php
// --- SESSION START ---
session_start();
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables
$user = [];
$currentUserId = getCurrentUserId();

try {
    $pdo = getDBConnection();
    
    // Get user data and account data
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // If user record doesn't exist, set defaults
    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => 'your-email@example.com', 'account_type' => 'personal'];
    }
    
    // Check if user has premium card
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ? AND status = 'active'");
    $stmt->execute([$currentUserId]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get all transactions for the user
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $allTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get pending transactions
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? AND status = 'pending' ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $pendingTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get completed transactions
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? AND status = 'completed' ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $completedTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get failed transactions
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? AND status = 'failed' ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $failedTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate statistics
    $totalTransactions = count($allTransactions);
    $pendingCount = count($pendingTransactions);
    $completedCount = count($completedTransactions);
    $failedCount = count($failedTransactions);
    
    // Calculate total amounts
    $totalDeposited = 0;
    $totalTransferred = 0;
    foreach ($allTransactions as $transaction) {
        if ($transaction['type'] === 'credit' && $transaction['status'] === 'completed') {
            $totalDeposited += $transaction['amount'];
        } elseif ($transaction['type'] === 'debit' && $transaction['status'] === 'completed') {
            $totalTransferred += $transaction['amount'];
        }
    }

} catch (PDOException $e) {
    error_log("Track Money error: " . $e->getMessage());
    // Set sane defaults to prevent HTML errors
    $user = [
        'full_name' => 'User',
        'email' => 'your-email@example.com',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $premiumCard = null;
    $allTransactions = [];
    $pendingTransactions = [];
    $completedTransactions = [];
    $failedTransactions = [];
    $totalTransactions = 0;
    $pendingCount = 0;
    $completedCount = 0;
    $failedCount = 0;
    $totalDeposited = 0;
    $totalTransferred = 0;
}

// Get user's first name
$firstName = 'User';
if (!empty($user['full_name']) && $user['full_name'] != 'User') {
    $nameParts = explode(' ', $user['full_name']);
    $firstName = $nameParts[0];
}

// Handle filter requests
$filter = $_GET['filter'] ?? 'all';
$search = $_GET['search'] ?? '';

// Filter transactions based on user selection
$displayTransactions = $allTransactions;
if ($filter === 'pending') {
    $displayTransactions = $pendingTransactions;
} elseif ($filter === 'completed') {
    $displayTransactions = $completedTransactions;
} elseif ($filter === 'failed') {
    $displayTransactions = $failedTransactions;
} elseif ($filter === 'deposits') {
    $displayTransactions = array_filter($allTransactions, function($transaction) {
        return $transaction['type'] === 'credit';
    });
} elseif ($filter === 'transfers') {
    $displayTransactions = array_filter($allTransactions, function($transaction) {
        return $transaction['type'] === 'debit';
    });
}

// Apply search filter if provided
if (!empty($search)) {
    $displayTransactions = array_filter($displayTransactions, function($transaction) use ($search) {
        return stripos($transaction['description'], $search) !== false || 
               stripos($transaction['recipient_account'], $search) !== false ||
               stripos($transaction['amount'], $search) !== false;
    });
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Track Money - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #facc15 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Status badges */
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-pending {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-completed {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-failed {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        /* Transaction type indicators */
        .transaction-type-credit {
            border-left: 4px solid #10b981;
        }
        
        .transaction-type-debit {
            border-left: 4px solid #ef4444;
        }
        
        /* Progress bars */
        .progress-bar {
            height: 8px;
            border-radius: 4px;
            background-color: #e5e7eb;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            border-radius: 4px;
            transition: width 0.5s ease-in-out;
        }
        
        /* Filter buttons */
        .filter-btn {
            transition: all 0.3s ease;
            border-radius: 12px;
        }
        
        .filter-btn.active {
            background-color: #22c55e;
            color: white;
        }
        
        /* Loading skeleton */
        .skeleton {
            background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
            background-size: 200% 100%;
            animation: loading 1.5s infinite;
        }
        
        @keyframes loading {
            0% {
                background-position: 200% 0;
            }
            100% {
                background-position: -200% 0;
            }
        }
        
        /* Chart container */
        .chart-container {
            position: relative;
            height: 200px;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-chart-line text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Track Your Money 📊</h1>
                <p class="text-gray-600 mt-1 font-medium">Monitor your deposits, transfers, and transaction status in real-time.</p>
            </div>

            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <!-- Total Transactions -->
                <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Total Transactions</p>
                            <p class="text-2xl font-bold text-gray-900 mt-1"><?php echo $totalTransactions; ?></p>
                        </div>
                        <div class="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-blue-600 text-xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <div class="progress-bar">
                            <div class="progress-fill bg-blue-500" style="width: 100%"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Pending Transactions -->
                <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Pending</p>
                            <p class="text-2xl font-bold text-gray-900 mt-1"><?php echo $pendingCount; ?></p>
                        </div>
                        <div class="w-12 h-12 rounded-full bg-yellow-100 flex items-center justify-center">
                            <i class="fas fa-clock text-yellow-600 text-xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <div class="progress-bar">
                            <div class="progress-fill bg-yellow-500" style="width: <?php echo $totalTransactions > 0 ? ($pendingCount / $totalTransactions * 100) : 0; ?>%"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Completed Transactions -->
                <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Completed</p>
                            <p class="text-2xl font-bold text-gray-900 mt-1"><?php echo $completedCount; ?></p>
                        </div>
                        <div class="w-12 h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-check-circle text-green-600 text-xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <div class="progress-bar">
                            <div class="progress-fill bg-green-500" style="width: <?php echo $totalTransactions > 0 ? ($completedCount / $totalTransactions * 100) : 0; ?>%"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Failed Transactions -->
                <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Failed</p>
                            <p class="text-2xl font-bold text-gray-900 mt-1"><?php echo $failedCount; ?></p>
                        </div>
                        <div class="w-12 h-12 rounded-full bg-red-100 flex items-center justify-center">
                            <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <div class="progress-bar">
                            <div class="progress-fill bg-red-500" style="width: <?php echo $totalTransactions > 0 ? ($failedCount / $totalTransactions * 100) : 0; ?>%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Financial Overview -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
                <!-- Amount Summary -->
                <div class="lg:col-span-2 gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                        <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                        Financial Overview
                    </h2>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Total Deposited -->
                        <div class="bg-white rounded-xl p-4 border border-gray-200">
                            <div class="flex items-center">
                                <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-3">
                                    <i class="fas fa-arrow-down text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-medium text-gray-600">Total Deposited</p>
                                    <p class="text-lg font-bold text-gray-900">$<?php echo number_format($totalDeposited, 2); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Total Transferred -->
                        <div class="bg-white rounded-xl p-4 border border-gray-200">
                            <div class="flex items-center">
                                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-3">
                                    <i class="fas fa-arrow-up text-red-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-medium text-gray-600">Total Transferred</p>
                                    <p class="text-lg font-bold text-gray-900">$<?php echo number_format($totalTransferred, 2); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Current Balance -->
                        <div class="bg-white rounded-xl p-4 border border-gray-200 md:col-span-2">
                            <div class="flex items-center justify-between">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-wallet text-blue-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-medium text-gray-600">Current Balance</p>
                                        <p class="text-lg font-bold text-gray-900">$<?php echo number_format($user['balance'] ?? 0, 2); ?></p>
                                    </div>
                                </div>
                                <a href="deposit.php" class="bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-xl transition-colors duration-300">
                                    Add Funds
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Transaction Distribution -->
                <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                    <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                        <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                        Transaction Types
                    </h2>
                    
                    <div class="space-y-4">
                        <?php
                        $depositCount = count(array_filter($allTransactions, function($t) { return $t['type'] === 'credit'; }));
                        $transferCount = count(array_filter($allTransactions, function($t) { return $t['type'] === 'debit'; }));
                        $totalCount = $depositCount + $transferCount;
                        ?>
                        
                        <div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm font-medium text-gray-700">Deposits</span>
                                <span class="text-sm font-bold text-gray-900"><?php echo $depositCount; ?></span>
                            </div>
                            <div class="progress-bar">
                                <div class="progress-fill bg-green-500" style="width: <?php echo $totalCount > 0 ? ($depositCount / $totalCount * 100) : 0; ?>%"></div>
                            </div>
                        </div>
                        
                        <div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm font-medium text-gray-700">Transfers</span>
                                <span class="text-sm font-bold text-gray-900"><?php echo $transferCount; ?></span>
                            </div>
                            <div class="progress-bar">
                                <div class="progress-fill bg-red-500" style="width: <?php echo $totalCount > 0 ? ($transferCount / $totalCount * 100) : 0; ?>%"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mt-6 pt-4 border-t border-gray-200">
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-600">Total Transactions</span>
                            <span class="text-sm font-bold text-gray-900"><?php echo $totalCount; ?></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Transaction Tracking -->
            <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                    <h2 class="text-xl font-bold text-gray-900 flex items-center mb-4 md:mb-0">
                        <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                        Transaction Tracking
                    </h2>
                    
                    <!-- Filters and Search -->
                    <div class="flex flex-col sm:flex-row gap-3">
                        <!-- Search -->
                        <div class="relative">
                            <input type="text" id="search-input" placeholder="Search transactions..." class="form-input w-full sm:w-64 rounded-xl border border-gray-300 pl-10 pr-4 py-2 focus:outline-none" value="<?php echo htmlspecialchars($search); ?>">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                        </div>
                        
                        <!-- Filter Dropdown -->
                        <div class="relative">
                            <select id="filter-select" class="form-input w-full sm:w-48 rounded-xl border border-gray-300 pl-4 pr-10 py-2 focus:outline-none appearance-none">
                                <option value="all" <?php echo $filter === 'all' ? 'selected' : ''; ?>>All Transactions</option>
                                <option value="pending" <?php echo $filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="completed" <?php echo $filter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                                <option value="failed" <?php echo $filter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                                <option value="deposits" <?php echo $filter === 'deposits' ? 'selected' : ''; ?>>Deposits Only</option>
                                <option value="transfers" <?php echo $filter === 'transfers' ? 'selected' : ''; ?>>Transfers Only</option>
                            </select>
                            <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-chevron-down text-gray-400"></i>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Transaction List -->
                <div class="overflow-hidden rounded-xl border border-gray-200">
                    <?php if (!empty($displayTransactions)): ?>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    <?php foreach ($displayTransactions as $transaction): ?>
                                        <tr class="hover:bg-gray-50 transition-colors <?php echo $transaction['type'] === 'credit' ? 'transaction-type-credit' : 'transaction-type-debit'; ?>">
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="flex items-center">
                                                    <div class="w-10 h-10 rounded-full <?php echo $transaction['type'] === 'credit' ? 'bg-green-100' : 'bg-red-100'; ?> flex items-center justify-center mr-3">
                                                        <i class="<?php echo $transaction['type'] === 'credit' ? 'fas fa-arrow-down text-green-600' : 'fas fa-arrow-up text-red-600'; ?>"></i>
                                                    </div>
                                                    <div>
                                                        <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($transaction['description']); ?></div>
                                                        <?php if (!empty($transaction['recipient_account'])): ?>
                                                            <div class="text-xs text-gray-500">To: <?php echo htmlspecialchars($transaction['recipient_account']); ?></div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <span class="text-sm font-medium <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                    <?php echo ucfirst($transaction['type']); ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <span class="text-sm font-bold <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                    <?php echo $transaction['type'] === 'credit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                <?php echo date('M j, Y g:i A', strtotime($transaction['created_at'])); ?>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                                    <?php echo ucfirst($transaction['status']); ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <button class="text-green-600 hover:text-green-900 mr-3 view-details" data-id="<?php echo $transaction['id']; ?>">
                                                    <i class="fas fa-eye mr-1"></i> Details
                                                </button>
                                                <?php if ($transaction['status'] === 'pending'): ?>
                                                    <button class="text-red-600 hover:text-red-900 cancel-transaction" data-id="<?php echo $transaction['id']; ?>">
                                                        <i class="fas fa-times mr-1"></i> Cancel
                                                    </button>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-12">
                            <div class="w-16 h-16 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                <i class="fas fa-receipt text-gray-400 text-2xl"></i>
                            </div>
                            <p class="text-gray-500 font-medium">No transactions found</p>
                            <p class="text-sm text-gray-400 mt-1">Your transactions will appear here</p>
                            <div class="mt-4">
                                <a href="deposit.php" class="inline-block bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-xl transition-colors duration-300 mr-3">
                                    Make a Deposit
                                </a>
                                <a href="wire_transfer.php" class="inline-block bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-xl transition-colors duration-300">
                                    Send Money
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Pagination (if needed in future) -->
                <?php if (!empty($displayTransactions) && count($displayTransactions) > 10): ?>
                    <div class="mt-6 flex items-center justify-between">
                        <div class="text-sm text-gray-700">
                            Showing <span class="font-medium">1</span> to <span class="font-medium">10</span> of <span class="font-medium"><?php echo count($displayTransactions); ?></span> results
                        </div>
                        <div class="flex space-x-2">
                            <button class="px-3 py-1 rounded-lg border border-gray-300 text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                                Previous
                            </button>
                            <button class="px-3 py-1 rounded-lg border border-gray-300 text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                                Next
                            </button>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="deposit.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-wallet text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Deposit</span>
                </a>
                <a href="track_money.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                        <i class="fas fa-chart-line text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Track</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Profile</span>
                </a>
            </div>
        </div>
    </div>

    <!-- Transaction Details Modal -->
    <div id="transaction-details-modal" class="fixed inset-0 z-[60] flex items-center justify-center bg-black bg-opacity-60 hidden transition-opacity duration-300">
        <div class="bg-white rounded-xl shadow-2xl p-6 w-full max-w-md mx-4 animate-slide-down" style="animation-duration: 0.3s;">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-xl font-bold text-gray-900">Transaction Details</h2>
                <button id="close-details-modal-btn" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times text-2xl"></i>
                </button>
            </div>
            <div id="transaction-details-content" class="space-y-4">
                <!-- Details will be loaded here via JavaScript -->
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation();
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                document.addEventListener('click', function(e) {
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
            
            // Filter and Search Functionality
            const filterSelect = document.getElementById('filter-select');
            const searchInput = document.getElementById('search-input');
            
            function updateFilters() {
                const filter = filterSelect.value;
                const search = searchInput.value;
                
                let url = 'track_money.php?';
                if (filter !== 'all') {
                    url += 'filter=' + filter;
                }
                if (search) {
                    url += (url.includes('?') ? '&' : '') + 'search=' + encodeURIComponent(search);
                }
                
                window.location.href = url;
            }
            
            if (filterSelect) {
                filterSelect.addEventListener('change', updateFilters);
            }
            
            if (searchInput) {
                // Add debounce to search
                let searchTimeout;
                searchInput.addEventListener('input', function() {
                    clearTimeout(searchTimeout);
                    searchTimeout = setTimeout(updateFilters, 500);
                });
            }
            
            // Transaction Details Modal
            const detailsModal = document.getElementById('transaction-details-modal');
            const closeDetailsModalBtn = document.getElementById('close-details-modal-btn');
            const detailsContent = document.getElementById('transaction-details-content');
            const viewDetailsButtons = document.querySelectorAll('.view-details');
            
            if (closeDetailsModalBtn) {
                closeDetailsModalBtn.addEventListener('click', () => {
                    detailsModal.classList.add('hidden');
                });
            }
            
            // Close modal when clicking outside
            detailsModal.addEventListener('click', (e) => {
                if (e.target === detailsModal) {
                    detailsModal.classList.add('hidden');
                }
            });
            
            // View Details Button Handler
            viewDetailsButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const transactionId = this.getAttribute('data-id');
                    loadTransactionDetails(transactionId);
                });
            });
            
            function loadTransactionDetails(transactionId) {
                // In a real application, you would fetch this data from the server
                // For now, we'll simulate with the data we have
                
                // Find the transaction in the displayed transactions
                const transactionRow = document.querySelector(`.view-details[data-id="${transactionId}"]`).closest('tr');
                const description = transactionRow.querySelector('td:first-child .text-sm').textContent;
                const type = transactionRow.querySelector('td:nth-child(2) span').textContent;
                const amount = transactionRow.querySelector('td:nth-child(3) span').textContent;
                const date = transactionRow.querySelector('td:nth-child(4)').textContent;
                const status = transactionRow.querySelector('.status-badge').textContent;
                
                // Get recipient account if exists
                let recipient = '';
                const recipientElement = transactionRow.querySelector('td:first-child .text-xs');
                if (recipientElement) {
                    recipient = recipientElement.textContent.replace('To: ', '');
                }
                
                // Build details content
                detailsContent.innerHTML = `
                    <div class="space-y-3">
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Description:</span>
                            <span class="text-gray-900">${description}</span>
                        </div>
                        ${recipient ? `
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Recipient:</span>
                            <span class="text-gray-900">${recipient}</span>
                        </div>
                        ` : ''}
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Type:</span>
                            <span class="font-semibold ${type === 'Credit' ? 'text-green-600' : 'text-red-600'}">${type}</span>
                        </div>
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Amount:</span>
                            <span class="font-bold ${type === 'Credit' ? 'text-green-600' : 'text-red-600'}">${amount}</span>
                        </div>
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Date:</span>
                            <span class="text-gray-900">${date}</span>
                        </div>
                        <div class="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                            <span class="font-medium text-gray-700">Status:</span>
                            <span class="status-badge status-${status.toLowerCase()}">${status}</span>
                        </div>
                        ${status === 'Pending' ? `
                        <div class="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                            <p class="text-sm text-yellow-800">
                                <i class="fas fa-info-circle mr-1"></i>
                                This transaction is being processed. It usually takes 1-2 business days to complete.
                            </p>
                        </div>
                        ` : ''}
                    </div>
                `;
                
                detailsModal.classList.remove('hidden');
            }
            
            // Cancel Transaction Buttons
            const cancelButtons = document.querySelectorAll('.cancel-transaction');
            
            cancelButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const transactionId = this.getAttribute('data-id');
                    if (confirm('Are you sure you want to cancel this transaction? This action cannot be undone.')) {
                        // In a real application, you would send an AJAX request to cancel the transaction
                        alert('Cancellation request sent. Please wait for confirmation.');
                        // For demo purposes, we'll just reload the page
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    }
                });
            });
        });
    </script>

</body>
</html>