<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Handle fund management actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'manage_funds') {
        $user_id = $_POST['user_id'] ?? '';
        $amount = $_POST['amount'] ?? '';
        $type = $_POST['transaction_type'] ?? '';
        $description = $_POST['description'] ?? '';
        $currency = $_POST['currency'] ?? 'USD';
        $manual_date = $_POST['manual_date'] ?? '';
        $manual_time = $_POST['manual_time'] ?? '';
        
        if (!empty($user_id) && !empty($amount) && !empty($type)) {
            try {
                $pdo->beginTransaction();
                
                // Get user account
                $stmt = $pdo->prepare("SELECT * FROM accounts WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $account = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($account) {
                    // Convert amount if currency is different from account currency
                    $converted_amount = $amount;
                    if ($currency !== $account['currency']) {
                        $converted_amount = convertCurrency($amount, $currency, $account['currency']);
                    }
                    
                    if ($type === 'credit') {
                        // Add funds
                        $stmt = $pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE user_id = ?");
                        $stmt->execute([$converted_amount, $user_id]);
                        
                        $transaction_description = "Admin credit: " . $description . " (" . $currency . " " . number_format($amount, 2) . ")";
                    } else {
                        // Debit funds (check if sufficient balance)
                        if ($account['balance'] >= $converted_amount) {
                            $stmt = $pdo->prepare("UPDATE accounts SET balance = balance - ? WHERE user_id = ?");
                            $stmt->execute([$converted_amount, $user_id]);
                            
                            $transaction_description = "Admin debit: " . $description . " (" . $currency . " " . number_format($amount, 2) . ")";
                        } else {
                            throw new Exception("Insufficient funds");
                        }
                    }
                    
                    // Set transaction date (manual or current)
                    $transaction_date = date('Y-m-d H:i:s');
                    if (!empty($manual_date) && !empty($manual_time)) {
                        $transaction_date = $manual_date . ' ' . $manual_time . ':00';
                    }
                    
                    // Log transaction with manual date if provided
                    $stmt = $pdo->prepare("INSERT INTO transactions (user_id, type, amount, description, status, created_at) 
                                          VALUES (?, ?, ?, ?, 'completed', ?)");
                    $stmt->execute([$user_id, $type, $converted_amount, $transaction_description, $transaction_date]);
                    
                    // Create user notification
                    $stmt = $pdo->prepare("INSERT INTO user_notifications (user_id, title, message, type, sent_by) 
                                          VALUES (?, 'Account Update', ?, 'info', ?)");
                    $notification_message = "Your account has been {$type}ed with amount: " . $currency . " " . number_format($amount, 2) . ". " . $description;
                    $stmt->execute([$user_id, $notification_message, $admin_id]);
                    
                    $pdo->commit();
                    $_SESSION['admin_success'] = "Funds {$type}ed successfully!";
                } else {
                    throw new Exception("User account not found");
                }
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $_SESSION['admin_error'] = "Error: " . $e->getMessage();
            }
        }
    }
    
    header('Location: admin_funds.php');
    exit;
}

// Get all users for dropdown
try {
    $stmt = $pdo->prepare("SELECT u.id, u.full_name, u.email, u.phone, a.account_number, a.balance, a.currency 
                          FROM users u 
                          JOIN accounts a ON u.id = a.user_id 
                          WHERE u.is_active = 1 
                          ORDER BY u.full_name");
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get recent fund transactions
    $stmt = $pdo->prepare("SELECT t.*, u.full_name, u.email 
                          FROM transactions t 
                          JOIN users u ON t.user_id = u.id 
                          WHERE t.description LIKE 'Admin %' 
                          ORDER BY t.created_at DESC 
                          LIMIT 10");
    $stmt->execute();
    $recent_transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Admin Funds error: " . $e->getMessage());
    $users = [];
    $recent_transactions = [];
}

// Currency conversion function (simplified - in real app, use live rates)
function convertCurrency($amount, $from_currency, $to_currency) {
    // Simplified conversion rates - in production, use API like Open Exchange Rates
    $rates = [
        'USD' => 1.0,
        'EUR' => 0.85,
        'GBP' => 0.73,
        'AUD' => 1.35
    ];
    
    if ($from_currency === $to_currency) {
        return $amount;
    }
    
    // Convert to USD first, then to target currency
    $usd_amount = $amount / $rates[$from_currency];
    return $usd_amount * $rates[$to_currency];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fund Management - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-completed {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .transaction-type-credit {
            border-left: 4px solid #10b981;
        }
        
        .transaction-type-debit {
            border-left: 4px solid #ef4444;
        }
        
        .currency-flag {
            width: 20px;
            height: 15px;
            display: inline-block;
            margin-right: 8px;
            border-radius: 2px;
            background-size: cover;
        }
        
        .flag-usd { background: linear-gradient(to bottom, #b22234 33%, white 33%, white 66%, #b22234 66%); }
        .flag-eur { background-color: #003399; }
        .flag-gbp { background-color: #012169; }
        .flag-aud { background-color: #012169; }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .main-grid {
                grid-template-columns: 1fr !important;
            }
            
            .transaction-type-grid {
                grid-template-columns: 1fr !important;
                gap: 0.5rem !important;
            }
            
            .currency-amount-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
            }
            
            .datetime-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .transaction-card {
                padding: 1rem !important;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling improvements */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        .radio-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .radio-card.selected {
            border-color: #22c55e;
            background-color: #f0fdf4;
        }
        
        .radio-card.credit.selected {
            border-color: #10b981;
            background-color: #f0fdf4;
        }
        
        .radio-card.debit.selected {
            border-color: #ef4444;
            background-color: #fef2f2;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-money-bill-wave text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Fund Management 💰</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Add or remove funds from user accounts with multi-currency support</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 sm:gap-8 main-grid">
                <!-- Fund Management Form -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 sm:mb-6 flex items-center">
                        <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                        Manage User Funds
                    </h2>
                    
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="manage_funds">
                        
                        <div class="space-y-4 sm:space-y-6">
                            <!-- User Selection -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Select User</label>
                                <select name="user_id" required 
                                        class="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base"
                                        onchange="updateUserBalance(this)">
                                    <option value="">Choose a user...</option>
                                    <?php foreach ($users as $user): ?>
                                        <option value="<?php echo $user['id']; ?>" 
                                                data-balance="<?php echo $user['balance']; ?>"
                                                data-currency="<?php echo $user['currency']; ?>"
                                                data-account="<?php echo $user['account_number']; ?>">
                                            <?php echo htmlspecialchars($user['full_name']); ?> 
                                            (<?php echo htmlspecialchars($user['email']); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <div id="userBalance" class="text-xs sm:text-sm text-gray-500 mt-2 hidden">
                                    <div class="flex items-center justify-between">
                                        <span>Current Balance:</span>
                                        <span class="font-bold"><span id="balanceCurrency" class="font-medium"></span> <span id="balanceAmount">0.00</span></span>
                                    </div>
                                    <div id="accountNumber" class="text-gray-400 mt-1"></div>
                                </div>
                            </div>
                            
                            <!-- Transaction Type -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Transaction Type</label>
                                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4 transaction-type-grid">
                                    <label class="radio-card credit relative flex cursor-pointer border-2 border-gray-200 rounded-lg p-3 sm:p-4 hover:border-green-500 transition-colors">
                                        <input type="radio" name="transaction_type" value="credit" class="sr-only" required>
                                        <div class="flex items-center w-full">
                                            <i class="fas fa-plus-circle text-green-600 text-lg sm:text-xl mr-2 sm:mr-3"></i>
                                            <div>
                                                <span class="font-medium text-sm sm:text-base">Add Funds</span>
                                                <p class="text-xs text-gray-500 mt-1">Credit to user account</p>
                                            </div>
                                        </div>
                                    </label>
                                    
                                    <label class="radio-card debit relative flex cursor-pointer border-2 border-gray-200 rounded-lg p-3 sm:p-4 hover:border-red-500 transition-colors">
                                        <input type="radio" name="transaction_type" value="debit" class="sr-only" required>
                                        <div class="flex items-center w-full">
                                            <i class="fas fa-minus-circle text-red-600 text-lg sm:text-xl mr-2 sm:mr-3"></i>
                                            <div>
                                                <span class="font-medium text-sm sm:text-base">Remove Funds</span>
                                                <p class="text-xs text-gray-500 mt-1">Debit from user account</p>
                                            </div>
                                        </div>
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Currency and Amount -->
                            <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 currency-amount-grid">
                                <div class="sm:col-span-1">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Currency</label>
                                    <select name="currency" required 
                                            class="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                                        <option value="USD">USD ($)</option>
                                        <option value="EUR">EUR (€)</option>
                                        <option value="GBP">GBP (£)</option>
                                        <option value="AUD">AUD (A$)</option>
                                    </select>
                                </div>
                                <div class="sm:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Amount</label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span class="text-gray-500 font-medium text-sm sm:text-base" id="currencySymbol">$</span>
                                        </div>
                                        <input type="number" name="amount" step="0.01" min="0.01" required 
                                               class="pl-10 w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base"
                                               placeholder="0.00"
                                               oninput="updateConversionPreview()">
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Manual Date & Time -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Transaction Date & Time
                                    <span class="text-xs text-gray-500 font-normal">(Optional - for backdating)</span>
                                </label>
                                <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 datetime-grid">
                                    <div>
                                        <input type="date" name="manual_date" 
                                               class="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base"
                                               max="<?php echo date('Y-m-d'); ?>">
                                        <p class="text-xs text-gray-500 mt-1">Select date</p>
                                    </div>
                                    <div>
                                        <input type="time" name="manual_time" 
                                               class="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base"
                                               value="<?php echo date('H:i'); ?>">
                                        <p class="text-xs text-gray-500 mt-1">Select time</p>
                                    </div>
                                </div>
                                <div class="mt-2 p-3 bg-blue-50 rounded-lg">
                                    <div class="flex items-start">
                                        <i class="fas fa-info-circle text-blue-500 mt-0.5 mr-2 text-sm"></i>
                                        <div class="text-xs sm:text-sm text-blue-700">
                                            <p class="font-medium">Backdating Information</p>
                                            <p>Leave empty to use current date/time. Useful for correcting past transactions.</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Description -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                                <textarea name="description" rows="3" required 
                                          class="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base"
                                          placeholder="Reason for this transaction..."></textarea>
                            </div>
                            
                            <!-- Conversion Preview -->
                            <div id="conversionPreview" class="hidden p-3 sm:p-4 bg-gray-50 rounded-lg border border-gray-200">
                                <h4 class="font-medium text-gray-900 mb-2 text-sm sm:text-base">Currency Conversion Preview</h4>
                                <div class="text-xs sm:text-sm text-gray-600 space-y-1">
                                    <div class="flex justify-between">
                                        <span>Transaction Amount:</span>
                                        <span id="previewAmount" class="font-medium"></span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span>Converted Amount:</span>
                                        <span id="previewConverted" class="font-medium"></span>
                                    </div>
                                    <p class="text-xs text-gray-500 mt-2">Conversion rates are approximate and for reference only.</p>
                                </div>
                            </div>
                            
                            <!-- Submit Button -->
                            <div>
                                <button type="submit" 
                                        class="w-full bg-green-600 text-white py-3 px-6 rounded-lg hover:bg-green-700 transition-colors font-medium text-base sm:text-lg shadow-sm">
                                    <i class="fas fa-paper-plane mr-2"></i>
                                    Process Transaction
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Recent Transactions -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 sm:mb-6 flex items-center">
                        <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                        Recent Fund Transactions
                    </h2>
                    
                    <div class="space-y-3 sm:space-y-4">
                        <?php if (!empty($recent_transactions)): ?>
                            <?php foreach ($recent_transactions as $transaction): ?>
                                <div class="border border-gray-200 rounded-lg p-3 sm:p-4 hover:shadow-md transition-shadow transaction-card <?php echo $transaction['type'] === 'credit' ? 'transaction-type-credit' : 'transaction-type-debit'; ?>">
                                    <div class="flex justify-between items-start mb-2">
                                        <div class="flex-1 min-w-0">
                                            <p class="font-medium text-gray-900 text-sm sm:text-base truncate"><?php echo htmlspecialchars($transaction['full_name']); ?></p>
                                            <p class="text-xs sm:text-sm text-gray-500 truncate"><?php echo htmlspecialchars($transaction['email']); ?></p>
                                        </div>
                                        <span class="text-base sm:text-lg font-bold ml-2 flex-shrink-0 <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                            <?php echo $transaction['type'] === 'credit' ? '+' : '-'; ?>
                                            <?php 
                                                // Extract currency from description
                                                if (preg_match('/\(([A-Z]{3})\s([0-9.,]+)\)/', $transaction['description'], $matches)) {
                                                    echo $matches[1] . ' ' . $matches[2];
                                                } else {
                                                    echo '$' . number_format($transaction['amount'], 2);
                                                }
                                            ?>
                                        </span>
                                    </div>
                                    <p class="text-xs sm:text-sm text-gray-600 mb-2 line-clamp-2"><?php echo htmlspecialchars($transaction['description']); ?></p>
                                    <div class="flex justify-between items-center text-xs text-gray-500">
                                        <span class="flex items-center">
                                            <i class="fas fa-clock mr-1"></i>
                                            <?php echo date('M j, Y g:i A', strtotime($transaction['created_at'])); ?>
                                        </span>
                                        <span class="status-badge status-completed text-xs">Completed</span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="text-center py-6 sm:py-8">
                                <div class="w-12 h-12 sm:w-16 sm:h-16 mx-auto mb-3 sm:mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                    <i class="fas fa-exchange-alt text-gray-400 text-lg sm:text-xl"></i>
                                </div>
                                <p class="text-gray-500 font-medium text-sm sm:text-base">No recent fund transactions</p>
                                <p class="text-gray-400 text-xs sm:text-sm mt-1">Transactions will appear here once processed</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
            
            // Initialize radio button styling
            document.querySelectorAll('input[name="transaction_type"]').forEach(radio => {
                if (radio.checked) {
                    updateRadioCardStyle(radio);
                }
            });
        });

        function updateUserBalance(select) {
            const userBalance = document.getElementById('userBalance');
            const balanceAmount = document.getElementById('balanceAmount');
            const balanceCurrency = document.getElementById('balanceCurrency');
            const accountNumber = document.getElementById('accountNumber');
            
            if (select.value) {
                const selectedOption = select.options[select.selectedIndex];
                const balance = selectedOption.getAttribute('data-balance');
                const currency = selectedOption.getAttribute('data-currency');
                const account = selectedOption.getAttribute('data-account');
                
                balanceAmount.textContent = parseFloat(balance).toFixed(2);
                balanceCurrency.textContent = currency;
                accountNumber.textContent = 'Account: ' + account;
                userBalance.classList.remove('hidden');
                
                updateConversionPreview();
            } else {
                userBalance.classList.add('hidden');
                document.getElementById('conversionPreview').classList.add('hidden');
            }
        }

        function updateConversionPreview() {
            const selectedUser = document.querySelector('select[name="user_id"]');
            const currencySelect = document.querySelector('select[name="currency"]');
            const amountInput = document.querySelector('input[name="amount"]');
            const previewDiv = document.getElementById('conversionPreview');
            
            if (!selectedUser.value || !amountInput.value || parseFloat(amountInput.value) <= 0) {
                previewDiv.classList.add('hidden');
                return;
            }
            
            const userCurrency = selectedUser.options[selectedUser.selectedIndex].getAttribute('data-currency');
            const transactionCurrency = currencySelect.value;
            const amount = parseFloat(amountInput.value);
            
            if (userCurrency === transactionCurrency) {
                previewDiv.classList.add('hidden');
                return;
            }
            
            // Simplified conversion rates (same as PHP)
            const rates = {
                'USD': 1.0,
                'EUR': 0.85,
                'GBP': 0.73,
                'AUD': 1.35
            };
            
            const convertedAmount = (amount / rates[transactionCurrency]) * rates[userCurrency];
            
            document.getElementById('previewAmount').textContent = 
                transactionCurrency + ' ' + amount.toFixed(2);
            document.getElementById('previewConverted').textContent = 
                userCurrency + ' ' + convertedAmount.toFixed(2);
            
            previewDiv.classList.remove('hidden');
        }

        // Update currency symbol when currency changes
        document.querySelector('select[name="currency"]').addEventListener('change', function() {
            const symbols = {
                'USD': '$',
                'EUR': '€',
                'GBP': '£',
                'AUD': 'A$'
            };
            document.getElementById('currencySymbol').textContent = symbols[this.value];
            updateConversionPreview();
        });

        // Radio button styling
        document.querySelectorAll('input[name="transaction_type"]').forEach(radio => {
            radio.addEventListener('change', function() {
                updateRadioCardStyle(this);
            });
        });

        function updateRadioCardStyle(radio) {
            document.querySelectorAll('.radio-card').forEach(card => {
                card.classList.remove('selected', 'credit', 'debit');
                card.classList.add('border-gray-200');
            });
            
            const parent = radio.closest('.radio-card');
            parent.classList.add('selected');
            parent.classList.remove('border-gray-200');
            
            if (radio.value === 'credit') {
                parent.classList.add('border-green-500', 'credit');
            } else {
                parent.classList.add('border-red-500', 'debit');
            }
        }

        // Set max date for manual date to today
        document.querySelector('input[name="manual_date"]').max = new Date().toISOString().split('T')[0];
    </script>

</body>
</html>