<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Handle support ticket actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $ticket_id = $_POST['ticket_id'] ?? '';
    
    if ($action === 'update_status' && !empty($ticket_id)) {
        $new_status = $_POST['status'] ?? '';
        $assigned_to = $_POST['assigned_to'] ?? null;
        
        if (!empty($new_status)) {
            try {
                $stmt = $pdo->prepare("UPDATE support_tickets SET status = ?, assigned_to = ? WHERE id = ?");
                $stmt->execute([$new_status, $assigned_to, $ticket_id]);
                
                $_SESSION['admin_success'] = "Ticket status updated successfully!";
            } catch (Exception $e) {
                $_SESSION['admin_error'] = "Error updating ticket: " . $e->getMessage();
            }
        }
    }
    
    header('Location: admin_support.php');
    exit;
}

// Get all support tickets with user info
try {
    $status_filter = $_GET['status'] ?? '';
    $priority_filter = $_GET['priority'] ?? '';
    
    $query = "SELECT st.*, u.full_name, u.email, u.phone, 
                     a.username as assigned_admin
              FROM support_tickets st 
              JOIN users u ON st.user_id = u.id 
              LEFT JOIN admin_users a ON st.assigned_to = a.id 
              WHERE 1=1";
    
    $params = [];
    
    if (!empty($status_filter)) {
        $query .= " AND st.status = ?";
        $params[] = $status_filter;
    }
    
    if (!empty($priority_filter)) {
        $query .= " AND st.priority = ?";
        $params[] = $priority_filter;
    }
    
    $query .= " ORDER BY 
        CASE st.priority 
            WHEN 'urgent' THEN 1
            WHEN 'high' THEN 2
            WHEN 'medium' THEN 3
            WHEN 'low' THEN 4
        END, st.created_at DESC";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get ticket statistics
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'open' THEN 1 ELSE 0 END) as open,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved,
            SUM(CASE WHEN status = 'closed' THEN 1 ELSE 0 END) as closed
        FROM support_tickets
    ");
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get admin users for assignment
    $stmt = $pdo->query("SELECT id, username, full_name FROM admin_users WHERE is_active = 1");
    $admins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Admin Support error: " . $e->getMessage());
    $tickets = [];
    $stats = ['total' => 0, 'open' => 0, 'pending' => 0, 'resolved' => 0, 'closed' => 0];
    $admins = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Support Tickets -Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-open { background-color: #dbeafe; color: #1e40af; }
        .status-pending { background-color: #fef3c7; color: #d97706; }
        .status-resolved { background-color: #d1fae5; color: #065f46; }
        .status-closed { background-color: #e5e7eb; color: #374151; }
        
        .priority-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .priority-urgent { background-color: #fee2e2; color: #dc2626; }
        .priority-high { background-color: #fef3c7; color: #d97706; }
        .priority-medium { background-color: #dbeafe; color: #1e40af; }
        .priority-low { background-color: #e5e7eb; color: #374151; }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .filter-container {
                flex-direction: column;
                align-items: stretch;
            }
            
            .filter-fields {
                flex-direction: column;
                gap: 1rem;
            }
            
            .filter-actions {
                flex-direction: row;
                justify-content: space-between;
                margin-top: 1rem;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 1000px;
            }
            
            .ticket-card {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
            
            .filter-actions {
                flex-direction: column;
                gap: 0.75rem;
            }
            
            .filter-actions button {
                width: 100%;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling improvements */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-headset text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Support Tickets 🎫</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Manage customer support requests and inquiries</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Ticket Statistics -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 sm:gap-6 mb-6 sm:mb-8 stats-grid">
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-blue-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Tickets</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['total']; ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-ticket-alt text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-red-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Open</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['open']; ?></p>
                            <p class="text-xs text-red-600 font-medium mt-1">Requires attention</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-red-100 flex items-center justify-center">
                            <i class="fas fa-exclamation-circle text-red-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-yellow-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Pending</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['pending']; ?></p>
                            <p class="text-xs text-yellow-600 font-medium mt-1">Awaiting response</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-yellow-100 flex items-center justify-center">
                            <i class="fas fa-clock text-yellow-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-green-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Resolved</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['resolved']; ?></p>
                            <p class="text-xs text-green-600 font-medium mt-1">Completed</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-check-circle text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-gray-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Closed</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['closed']; ?></p>
                            <p class="text-xs text-gray-600 font-medium mt-1">Archived</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-gray-100 flex items-center justify-center">
                            <i class="fas fa-archive text-gray-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filter Section -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 animate-slide-up card-padding">
                <form method="GET" action="" class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 filter-container">
                    <div class="flex flex-col md:flex-row md:items-center space-y-4 md:space-y-0 md:space-x-4 filter-fields">
                        <div class="w-full md:w-auto">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <select name="status" class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                                <option value="">All Status</option>
                                <option value="open" <?php echo $status_filter === 'open' ? 'selected' : ''; ?>>Open</option>
                                <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="resolved" <?php echo $status_filter === 'resolved' ? 'selected' : ''; ?>>Resolved</option>
                                <option value="closed" <?php echo $status_filter === 'closed' ? 'selected' : ''; ?>>Closed</option>
                            </select>
                        </div>
                        <div class="w-full md:w-auto">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Priority</label>
                            <select name="priority" class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                                <option value="">All Priority</option>
                                <option value="urgent" <?php echo $priority_filter === 'urgent' ? 'selected' : ''; ?>>Urgent</option>
                                <option value="high" <?php echo $priority_filter === 'high' ? 'selected' : ''; ?>>High</option>
                                <option value="medium" <?php echo $priority_filter === 'medium' ? 'selected' : ''; ?>>Medium</option>
                                <option value="low" <?php echo $priority_filter === 'low' ? 'selected' : ''; ?>>Low</option>
                            </select>
                        </div>
                    </div>
                    <div class="flex space-x-4 filter-actions">
                        <button type="submit" class="bg-green-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base shadow-sm">
                            <i class="fas fa-filter mr-2"></i>
                            Apply Filters
                        </button>
                        <a href="admin_support.php" class="bg-gray-200 text-gray-700 px-4 sm:px-6 py-2 rounded-lg hover:bg-gray-300 transition-colors font-medium text-sm sm:text-base text-center">
                            <i class="fas fa-times mr-2"></i>
                            Clear Filters
                        </a>
                    </div>
                </form>
            </div>

            <!-- Tickets Count -->
            <?php if (!empty($tickets)): ?>
                <div class="mb-4 text-sm text-gray-500 animate-slide-up">
                    Showing <span class="font-semibold"><?php echo count($tickets); ?></span> ticket<?php echo count($tickets) !== 1 ? 's' : ''; ?>
                    <?php if (!empty($status_filter) || !empty($priority_filter)): ?>
                        matching your filters
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <!-- Tickets Table -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-slide-up">
                <?php if (!empty($tickets)): ?>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ticket</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Subject & Category</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Priority</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Assigned To</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($tickets as $ticket): ?>
                                    <tr class="hover:bg-gray-50 transition-colors">
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900">#TK<?php echo str_pad($ticket['id'], 6, '0', STR_PAD_LEFT); ?></div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900 truncate max-w-[120px]"><?php echo htmlspecialchars($ticket['full_name']); ?></div>
                                            <div class="text-sm text-gray-500 truncate max-w-[120px]"><?php echo htmlspecialchars($ticket['email']); ?></div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4">
                                            <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($ticket['subject']); ?></div>
                                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($ticket['category']); ?></div>
                                            <div class="text-xs text-gray-400 mt-1 truncate max-w-[200px]">
                                                <?php echo htmlspecialchars(substr($ticket['message'], 0, 80)); ?>...
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="priority-badge priority-<?php echo $ticket['priority']; ?>">
                                                <?php echo ucfirst($ticket['priority']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="status-badge status-<?php echo $ticket['status']; ?>">
                                                <?php echo ucfirst($ticket['status']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <?php echo $ticket['assigned_admin'] ? htmlspecialchars($ticket['assigned_admin']) : '<span class="text-gray-400">Unassigned</span>'; ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <div class="flex flex-col">
                                                <span><?php echo date('M j, Y', strtotime($ticket['created_at'])); ?></span>
                                                <span class="text-xs text-gray-400"><?php echo date('g:i A', strtotime($ticket['created_at'])); ?></span>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm font-medium">
                                            <div class="flex flex-col space-y-2">
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                                                    <input type="hidden" name="action" value="update_status">
                                                    <select name="status" onchange="this.form.submit()" class="text-xs border border-gray-300 rounded px-2 py-1 w-full focus:ring-1 focus:ring-green-500 focus:border-green-500">
                                                        <option value="open" <?php echo $ticket['status'] === 'open' ? 'selected' : ''; ?>>Open</option>
                                                        <option value="pending" <?php echo $ticket['status'] === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                                        <option value="resolved" <?php echo $ticket['status'] === 'resolved' ? 'selected' : ''; ?>>Resolved</option>
                                                        <option value="closed" <?php echo $ticket['status'] === 'closed' ? 'selected' : ''; ?>>Closed</option>
                                                    </select>
                                                </form>
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                                                    <input type="hidden" name="action" value="update_status">
                                                    <select name="assigned_to" onchange="this.form.submit()" class="text-xs border border-gray-300 rounded px-2 py-1 w-full focus:ring-1 focus:ring-green-500 focus:border-green-500">
                                                        <option value="">Unassigned</option>
                                                        <?php foreach ($admins as $admin): ?>
                                                            <option value="<?php echo $admin['id']; ?>" <?php echo $ticket['assigned_to'] == $admin['id'] ? 'selected' : ''; ?>>
                                                                <?php echo htmlspecialchars($admin['username']); ?>
                                                            </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="w-16 h-16 sm:w-20 sm:h-20 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-ticket-alt text-gray-400 text-xl sm:text-2xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium text-lg sm:text-xl mb-2">No support tickets found</p>
                        <p class="text-gray-400 text-sm sm:text-base max-w-md mx-auto">
                            <?php echo (!empty($status_filter) || !empty($priority_filter)) ? 
                                'Try adjusting your search filters' : 
                                'No support tickets have been submitted yet'; ?>
                        </p>
                        <?php if (!empty($status_filter) || !empty($priority_filter)): ?>
                            <a href="admin_support.php" class="inline-block mt-4 bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm">
                                Clear All Filters
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Quick Actions -->
            <?php if (!empty($tickets)): ?>
                <div class="mt-6 bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-4 sm:p-6 border border-blue-100 animate-slide-up">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                        <i class="fas fa-bolt text-blue-600 mr-2"></i>
                        Quick Actions
                    </h3>
                    <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 text-sm">
                        <a href="#" class="flex items-center p-3 bg-white rounded-lg border border-blue-200 hover:border-blue-500 transition-colors">
                            <div class="w-8 h-8 rounded-full bg-blue-100 flex items-center justify-center mr-3">
                                <i class="fas fa-plus text-blue-600"></i>
                            </div>
                            <div>
                                <p class="font-medium text-gray-900">New Ticket</p>
                                <p class="text-xs text-gray-500">Create manual ticket</p>
                            </div>
                        </a>
                        <a href="#" class="flex items-center p-3 bg-white rounded-lg border border-green-200 hover:border-green-500 transition-colors">
                            <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center mr-3">
                                <i class="fas fa-download text-green-600"></i>
                            </div>
                            <div>
                                <p class="font-medium text-gray-900">Export Data</p>
                                <p class="text-xs text-gray-500">Download tickets</p>
                            </div>
                        </a>
                        <a href="#" class="flex items-center p-3 bg-white rounded-lg border border-purple-200 hover:border-purple-500 transition-colors">
                            <div class="w-8 h-8 rounded-full bg-purple-100 flex items-center justify-center mr-3">
                                <i class="fas fa-cog text-purple-600"></i>
                            </div>
                            <div>
                                <p class="font-medium text-gray-900">Settings</p>
                                <p class="text-xs text-gray-500">Support settings</p>
                            </div>
                        </a>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });
    </script>

</body>
</html>