<?php
session_start();
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

$currentUserId = getCurrentUserId();
$success_message = '';
$error_message = '';

try {
    $pdo = getDBConnection();
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        throw new Exception("User not found");
    }
    
    // Get user profile data if exists
    $stmt = $pdo->prepare("SELECT * FROM user_profiles WHERE user_id = ?");
    $stmt->execute([$currentUserId]);
    $profile = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Handle form submissions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['update_profile'])) {
            // Update basic profile information
            $full_name = trim($_POST['full_name']);
            $phone = trim($_POST['phone']);
            
            if (!empty($full_name)) {
                $stmt = $pdo->prepare("UPDATE users SET full_name = ?, phone = ? WHERE id = ?");
                $stmt->execute([$full_name, $phone, $currentUserId]);
                $success_message = "Profile updated successfully!";
                
                // Refresh user data
                $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                $stmt->execute([$currentUserId]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
            }
        }
        
        elseif (isset($_POST['update_password'])) {
            // Update password
            $current_password = $_POST['current_password'];
            $new_password = $_POST['new_password'];
            $confirm_password = $_POST['confirm_password'];
            
            if (password_verify($current_password, $user['password'])) {
                if ($new_password === $confirm_password) {
                    if (strlen($new_password) >= 6) {
                        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                        $stmt->execute([$hashed_password, $currentUserId]);
                        $success_message = "Password updated successfully!";
                    } else {
                        $error_message = "Password must be at least 6 characters long";
                    }
                } else {
                    $error_message = "New passwords do not match";
                }
            } else {
                $error_message = "Current password is incorrect";
            }
        }
        
        elseif (isset($_POST['update_profile_details'])) {
            // Update detailed profile information
            $date_of_birth = $_POST['date_of_birth'] ?: null;
            $address = trim($_POST['address']);
            $city = trim($_POST['city']);
            $state = trim($_POST['state']);
            $zip_code = trim($_POST['zip_code']);
            $country = trim($_POST['country']);
            $occupation = trim($_POST['occupation']);
            $annual_income = $_POST['annual_income'] ?: null;
            
            if ($profile) {
                // Update existing profile
                $stmt = $pdo->prepare("UPDATE user_profiles SET date_of_birth = ?, address = ?, city = ?, state = ?, zip_code = ?, country = ?, occupation = ?, annual_income = ? WHERE user_id = ?");
                $stmt->execute([$date_of_birth, $address, $city, $state, $zip_code, $country, $occupation, $annual_income, $currentUserId]);
            } else {
                // Create new profile
                $stmt = $pdo->prepare("INSERT INTO user_profiles (user_id, date_of_birth, address, city, state, zip_code, country, occupation, annual_income) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$currentUserId, $date_of_birth, $address, $city, $state, $zip_code, $country, $occupation, $annual_income]);
            }
            $success_message = "Profile details updated successfully!";
            
            // Refresh profile data
            $stmt = $pdo->prepare("SELECT * FROM user_profiles WHERE user_id = ?");
            $stmt->execute([$currentUserId]);
            $profile = $stmt->fetch(PDO::FETCH_ASSOC);
        }
    }
    
} catch (Exception $e) {
    error_log("Profile error: " . $e->getMessage());
    $error_message = "An error occurred while updating your profile";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - Oxland Credits</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">OXLAND</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name']); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
                <li>
                    <a href="profile.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-user-circle text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Profile</span>
                    </a>
                </li>
            </ul>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-8">
        <div class="max-w-4xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Page Header -->
            <div class="mb-6">
                <h1 class="text-3xl font-bold text-gray-900">Profile Settings 👤</h1>
                <p class="text-gray-600 mt-1">Manage your account information and preferences</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if ($success_message): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center">
                    <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-4">
                        <i class="fas fa-check text-green-600"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800">Success!</p>
                        <p class="text-sm text-green-700"><?php echo htmlspecialchars($success_message); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center">
                    <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800">Error</p>
                        <p class="text-sm text-red-700"><?php echo htmlspecialchars($error_message); ?></p>
                    </div>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Left Column - Profile Overview -->
                <div class="lg:col-span-1">
                    <!-- Profile Card -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm text-center">
                        <div class="w-20 h-20 mx-auto mb-4 rounded-full bg-green-100 flex items-center justify-center text-2xl font-bold text-green-700">
                            <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                        </div>
                        <h3 class="text-xl font-bold text-gray-900"><?php echo htmlspecialchars($user['full_name']); ?></h3>
                        <p class="text-gray-600 mt-1"><?php echo htmlspecialchars($user['email']); ?></p>
                        <div class="mt-4 p-3 bg-green-50 rounded-lg">
                            <p class="text-sm font-medium text-green-800">Account Verified</p>
                            <p class="text-xs text-green-600 mt-1">
                                <i class="fas fa-check-circle mr-1"></i>
                                Email verified
                            </p>
                        </div>
                    </div>

                    <!-- Account Info -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm mt-6">
                        <h4 class="font-bold text-gray-900 mb-4">Account Information</h4>
                        <div class="space-y-3">
                            <div>
                                <p class="text-xs text-gray-500">Member Since</p>
                                <p class="text-sm font-medium text-gray-900">
                                    <?php echo date('F j, Y', strtotime($user['created_at'])); ?>
                                </p>
                            </div>
                            <div>
                                <p class="text-xs text-gray-500">Account Type</p>
                                <p class="text-sm font-medium text-gray-900 capitalize">
                                    <?php echo htmlspecialchars($user['account_type']); ?>
                                </p>
                            </div>
                            <div>
                                <p class="text-xs text-gray-500">Last Login</p>
                                <p class="text-sm font-medium text-gray-900">
                                    <?php echo $user['last_login'] ? date('M j, Y g:i A', strtotime($user['last_login'])) : 'Never'; ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Right Column - Forms -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Basic Information Form -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm">
                        <h3 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Basic Information
                        </h3>
                        <form method="POST" action="">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Full Name</label>
                                    <input type="text" name="full_name" value="<?php echo htmlspecialchars($user['full_name']); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500" required>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                    <input type="email" value="<?php echo htmlspecialchars($user['email']); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg bg-gray-100" disabled>
                                    <p class="text-xs text-gray-500 mt-1">Contact support to change email</p>
                                </div>
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                                    <input type="tel" name="phone" value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                            </div>
                            <div class="mt-6">
                                <button type="submit" name="update_profile" 
                                        class="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium">
                                    Update Profile
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Change Password Form -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm">
                        <h3 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Change Password
                        </h3>
                        <form method="POST" action="">
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Current Password</label>
                                    <input type="password" name="current_password" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500" required>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">New Password</label>
                                    <input type="password" name="new_password" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500" required>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Confirm New Password</label>
                                    <input type="password" name="confirm_password" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500" required>
                                </div>
                            </div>
                            <div class="mt-6">
                                <button type="submit" name="update_password" 
                                        class="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium">
                                    Change Password
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Additional Profile Details -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm">
                        <h3 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Additional Information
                        </h3>
                        <form method="POST" action="">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Date of Birth</label>
                                    <input type="date" name="date_of_birth" value="<?php echo htmlspecialchars($profile['date_of_birth'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Occupation</label>
                                    <input type="text" name="occupation" value="<?php echo htmlspecialchars($profile['occupation'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Address</label>
                                    <input type="text" name="address" value="<?php echo htmlspecialchars($profile['address'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">City</label>
                                    <input type="text" name="city" value="<?php echo htmlspecialchars($profile['city'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">State/Province</label>
                                    <input type="text" name="state" value="<?php echo htmlspecialchars($profile['state'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">ZIP/Postal Code</label>
                                    <input type="text" name="zip_code" value="<?php echo htmlspecialchars($profile['zip_code'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Country</label>
                                    <input type="text" name="country" value="<?php echo htmlspecialchars($profile['country'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                </div>
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Annual Income (USD)</label>
                                    <input type="number" name="annual_income" value="<?php echo htmlspecialchars($profile['annual_income'] ?? ''); ?>" 
                                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500"
                                           placeholder="Optional">
                                </div>
                            </div>
                            <div class="mt-6">
                                <button type="submit" name="update_profile_details" 
                                        class="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium">
                                    Save Details
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="premiumcard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-credit-card text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Cards</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">History</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Profile</span>
                </a>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation();
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                document.addEventListener('click', function(e) {
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
        });
    </script>

</body>
</html>