<?php
session_start();
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables
$user = [];
$transactions = [];
$currentUserId = getCurrentUserId();

try {
    $pdo = getDBConnection();
    
    // Get user data and account data
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => 'your-email@example.com', 'account_type' => 'personal'];
    }
    
    // --- FETCH TRANSACTIONS ---
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);


} catch (PDOException $e) {
    error_log("Transaction Page error: " . $e->getMessage());
    $user = [
        'full_name' => 'User',
        'email' => 'your-email@example.com',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $transactions = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction History -Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-receipt text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Transaction History 🧾</h1>
                <p class="text-gray-600 mt-1 font-medium">View all your past account activity.</p>
            </div>

            <!-- Main Content Layout (Single Column) -->
            <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-down">
                <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                    <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                    Recent Activity
                </h2>

                <?php if (empty($transactions)): ?>
                    <!-- No Transactions Message -->
                    <div class="text-center py-12">
                        <div class="w-20 h-20 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-file-invoice-dollar text-gray-400 text-3xl"></i>
                        </div>
                        <p class="text-gray-600 font-semibold text-lg">No transactions found</p>
                        <p class="text-sm text-gray-500 mt-1">Your transaction history will appear here as you use your account.</p>
                        <a href="wire_transfer.php" class="inline-block mt-4 bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-5 rounded-xl transition-colors duration-300">
                            Make a Transfer
                        </a>
                    </div>
                <?php else: ?>
                    <!-- Transactions List -->
                    <div class="space-y-4">
                        <?php foreach ($transactions as $tx): ?>
                            <?php
                                // Determine icon and amount color
                                $isDebit = $tx['type'] == 'debit';
                                $iconClass = $isDebit ? 'fa-arrow-up text-red-600' : 'fa-arrow-down text-green-600';
                                $amountSign = $isDebit ? '-' : '+';
                                $amountClass = $isDebit ? 'text-red-700' : 'text-green-700';

                                // Format description
                                $description = htmlspecialchars($tx['description']);
                                if ($isDebit && !empty($tx['recipient_account'])) {
                                    $description = "To: " . htmlspecialchars($tx['recipient_account']);
                                } elseif ($tx['type'] == 'deposit') {
                                    $description = "Deposit";
                                } elseif (empty($description)) {
                                    $description = ucfirst($tx['type']);
                                }
                            ?>
                            <div class="flex items-center justify-between p-4 rounded-xl bg-white border border-gray-200 shadow-sm">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-full <?php echo $isDebit ? 'bg-red-50' : 'bg-green-50'; ?> flex items-center justify-center mr-4">
                                        <i class="fas <?php echo $iconClass; ?>"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900"><?php echo $description; ?></p>
                                        <p class="text-xs text-gray-500 font-medium">
                                            <?php echo date('M d, Y - h:i A', strtotime($tx['created_at'])); ?>
                                        </p>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <p class="text-sm font-bold <?php echo $amountClass; ?>">
                                        <?php echo $amountSign; ?>$<?php echo number_format($tx['amount'], 2); ?>
                                    </p>
                                    <p class="text-xs text-gray-500 font-medium capitalize">
                                        <?php echo htmlspecialchars($tx['status']); ?>
                                    </p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="premiumcard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-credit-card text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Cards</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">History</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Profile</span>
                </a>
            </div>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation(); // Prevent click from bubbling up
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                // Click outside to close
                document.addEventListener('click', function(e) {
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
        });
    </script>

</body>
</html>