<?php
require_once 'config.php';

$errors = [];
$success_message = '';

// Get user ID from session. If it's not there, they can't verify.
if (empty($_SESSION['otp_user_id'])) {
    // Redirect to login if they accessed this page directly
    redirect('login.php');
}

// Get user email for display
$user_email = $_SESSION['otp_user_email'] ?? 'your email';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = "Security token invalid. Please try again.";
    }

    // Sanitize OTP input
    $otp_code = sanitizeInput($_POST['otp_code'] ?? '');

    if (empty($otp_code)) {
        $errors[] = "OTP code is required.";
    }

    if (empty($errors)) {
        try {
            $pdo = getDBConnection();
            $user_id = $_SESSION['otp_user_id'];
            
            // --- USE THE NEW, CORRECT OTP VALIDATION FUNCTION ---
            $token_data = validateLoginOTP($otp_code, $user_id);
            
            if ($token_data) {
                // SUCCESS! OTP is correct.
                
                // Get full user data to set session
                $stmt = $pdo->prepare("SELECT id, email, full_name FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);

                // Update last login
                $update_stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                $update_stmt->execute([$user_id]);

                // --- FIX FOR REDIRECT LOOP ---
                // Force save the session *before* setting new data and redirecting.
                // This is also a security best practice.
                session_regenerate_id(true); 
                
                // Set final login session variables
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_email'] = $user['email'];
                $_SESSION['user_name'] = $user['full_name'];
                $_SESSION['logged_in'] = true;

                // Clear temporary OTP session data
                unset($_SESSION['otp_user_id']);
                unset($_SESSION['otp_user_email']);

                // Redirect to dashboard
                redirect('dashboard.php');
                exit;

            } else {
                // Failed login attempt
                $errors[] = "Invalid or expired OTP code. Please try again.";
                
                // Log this failed attempt
                $stmt = $pdo->prepare("INSERT INTO login_attempts (email, ip_address, success, method) VALUES (?, ?, ?, ?)");
                $stmt->execute([
                    $_SESSION['otp_user_email'] ?? 'unknown', 
                    $_SERVER['REMOTE_ADDR'], 
                    false, 
                    'otp'
                ]);
            }
        } catch (PDOException $e) {
            $errors[] = "A database error occurred. Please try again.";
            error_log("OTP Verification Error: " . $e->getMessage());
        }
    }
}

// Check for resend success message
if (isset($_SESSION['otp_resent'])) {
    $success_message = "A new OTP has been sent to your email.";
    unset($_SESSION['otp_resent']);
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verify Login - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-blue: #0d3b5d;
            --secondary-blue: #1e4e75;
            --accent-blue: #2c5f8a;
        }
        
        body { 
            font-family: 'Inter', sans-serif; 
            background: linear-gradient(135deg, #0d3b5d 0%, #1e4e75 50%, #2c5f8a 100%);
            min-height: 100vh;
            position: relative;
            overflow-x: hidden;
        }
        
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1200 800" opacity="0.03"><path fill="%23ffffff" d="M0,0H1200V800H0Z"/><circle cx="200" cy="200" r="150" fill="%23ffffff"/><circle cx="800" cy="600" r="120" fill="%23ffffff"/><rect x="400" y="100" width="300" height="200" fill="%23ffffff"/></svg>');
            background-size: cover;
            background-position: center;
            pointer-events: none;
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .floating-element {
            animation: float 6s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        .senior-illustration {
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 200 200" opacity="0.1"><path fill="%23ffffff" d="M100,50c27.61,0,50,22.39,50,50s-22.39,50-50,50S50,127.61,50,100,72.39,50,100,50Zm0,10c-22.09,0-40,17.91-40,40s17.91,40,40,40,40-17.91,40-40S122.09,60,100,60Z"/><circle cx="85" cy="85" r="5" fill="%23ffffff"/><circle cx="115" cy="85" r="5" fill="%23ffffff"/><path fill="%23ffffff" d="M80,115c0-11.05,8.95-20,20-20s20,8.95,20,20Z"/></svg>');
            background-size: contain;
            background-repeat: no-repeat;
            background-position: center;
        }
        
        .logo-circle {
            width: 120px;
            height: 120px;
            background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1), 0 0 0 1px rgba(255, 255, 255, 0.2);
            border: 4px solid rgba(255, 255, 255, 0.3);
        }
        
        .logo-image {
            width: 70px;
            height: 70px;
            object-fit: contain;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.2));
        }
        
        .otp-input {
            letter-spacing: 0.5em;
            font-weight: 600;
            font-size: 1.5rem;
        }
    </style>
</head>
<body class="flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <!-- Background Elements -->
    <div class="absolute inset-0 overflow-hidden">
        <div class="absolute -top-40 -right-32 w-80 h-80 bg-white rounded-full opacity-10 floating-element"></div>
        <div class="absolute top-1/2 -left-20 w-60 h-60 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -2s;"></div>
        <div class="absolute bottom-20 right-1/4 w-40 h-40 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -4s;"></div>
        
        <!-- Senior Illustration Elements -->
        <div class="absolute top-20 left-10 w-32 h-32 senior-illustration opacity-20 floating-element"></div>
        <div class="absolute bottom-32 right-20 w-24 h-24 senior-illustration opacity-20 floating-element" style="animation-delay: -3s;"></div>
        <div class="absolute top-1/3 right-1/4 w-28 h-28 senior-illustration opacity-20 floating-element" style="animation-delay: -1.5s;"></div>
    </div>

    <div class="max-w-md w-full space-y-8 relative z-10">
        <!-- Header -->
        <div class="text-center">
            <!-- Logo with Circular Background -->
            <div class="flex justify-center mb-6">
                <div class="logo-circle floating-element">
                    <img src="Uploads/images/logo.png" alt="Airwaves Credit Union" class="logo-image">
                </div>
            </div>
            <h2 class="text-center text-3xl font-extrabold text-white">
                Check Your Email
            </h2>
            <p class="mt-2 text-center text-lg text-blue-100">
                We sent a 6-digit verification code to <strong class="font-medium text-white"><?php echo htmlspecialchars($user_email); ?></strong>
            </p>
        </div>

        <!-- Success Message -->
        <?php if (!empty($success_message)): ?>
            <div class="glass-effect rounded-2xl p-4 border-l-4 border-green-500">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-check-circle text-green-500 text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm font-medium text-gray-800">
                            <?php echo htmlspecialchars($success_message); ?>
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <div class="glass-effect rounded-2xl p-4 border-l-4 border-red-500">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-circle text-red-500 text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <div class="mt-2 text-sm text-gray-700">
                            <ul class="list-disc list-inside space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- OTP Form -->
        <form class="glass-effect rounded-3xl p-8 space-y-6" method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div>
                <label for="otp_code" class="block text-sm font-semibold text-gray-700 mb-2">
                    <i class="fas fa-shield-alt text-[#0d3b5d] mr-2"></i>
                    Verification Code
                </label>
                <input id="otp_code" name="otp_code" type="text" inputmode="numeric" pattern="\d{6}" maxlength="6" required 
                       class="otp-input w-full px-4 py-4 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent text-center bg-white/80 transition-all duration-300"
                       placeholder="------"
                       autofocus>
            </div>

            <!-- Submit Button -->
            <div>
                <button type="submit" 
                        class="group relative w-full flex justify-center py-4 px-4 border border-transparent text-lg font-semibold rounded-xl text-white bg-[#0d3b5d] hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#0d3b5d] transition-all duration-300 transform hover:scale-[1.02] shadow-lg">
                    <i class="fas fa-shield-alt mr-3"></i>
                    Verify & Sign In
                </button>
            </div>

            <!-- Resend/Cancel Links -->
            <div class="text-center text-sm text-gray-600 space-y-2">
                <p class="font-medium">
                    Didn't receive the code? 
                    <a href="resend-otp.php" class="font-bold text-[#0d3b5d] hover:text-blue-800 transition-colors duration-300">
                        Resend code
                    </a>
                </p>
                <p>
                    <a href="logout.php" class="font-medium text-gray-500 hover:text-gray-700 transition-colors duration-300">
                        Cancel and go back
                    </a>
                </p>
            </div>
        </form>

        <!-- Security Notice -->
        <div class="text-center">
            <div class="inline-flex items-center space-x-2 text-xs text-blue-200 bg-blue-900/30 px-4 py-2 rounded-full">
                <i class="fas fa-clock"></i>
                <span>Code expires in 5 minutes</span>
            </div>
        </div>

        <!-- Senior-Friendly Features -->
        <div class="glass-effect rounded-2xl p-6 mt-6">
            <div class="flex items-center justify-between">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-[#0d3b5d] rounded-full flex items-center justify-center">
                        <i class="fas fa-lock text-white text-sm"></i>
                    </div>
                    <div>
                        <p class="text-sm font-semibold text-gray-800">Secure Verification</p>
                        <p class="text-xs text-gray-600">Two-factor authentication enabled</p>
                    </div>
                </div>
                <div class="text-2xl">🔒</div>
            </div>
        </div>
    </div>

    <script>
        // Enhanced OTP input functionality
        document.addEventListener('DOMContentLoaded', function() {
            const otpInput = document.getElementById('otp_code');
            
            if (otpInput) {
                // Auto-focus and select all text
                otpInput.focus();
                otpInput.select();
                
                // Auto-tab to next input (if we had multiple inputs)
                otpInput.addEventListener('input', function() {
                    if (this.value.length === 6) {
                        // Automatically submit form when 6 digits are entered
                        this.form.submit();
                    }
                });
                
                // Only allow numbers
                otpInput.addEventListener('keypress', function(e) {
                    if (!/^\d$/.test(e.key)) {
                        e.preventDefault();
                    }
                });
                
                // Paste event to handle OTP pasting
                otpInput.addEventListener('paste', function(e) {
                    e.preventDefault();
                    const pastedData = e.clipboardData.getData('text');
                    const numbersOnly = pastedData.replace(/\D/g, '');
                    this.value = numbersOnly.substring(0, 6);
                    
                    if (this.value.length === 6) {
                        this.form.submit();
                    }
                });
                
                // Focus styling
                otpInput.addEventListener('focus', function() {
                    this.classList.add('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
                
                otpInput.addEventListener('blur', function() {
                    this.classList.remove('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
            }
            
            // Add subtle background animation
            const floatingElements = document.querySelectorAll('.floating-element');
            floatingElements.forEach((element, index) => {
                element.style.animationDelay = `${-index * 2}s`;
            });
        });
    </script>
</body>
</html>