<?php
require_once 'config.php';

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitizeInput($_POST['email'] ?? '');
    
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Please enter a valid email address.";
    } else {
        try {
            $pdo = getDBConnection();
            
            // Check if user exists
            $stmt = $pdo->prepare("SELECT id, full_name FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                // Check rate limiting
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM verification_tokens WHERE user_id = ? AND token_type = 'password_reset' AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
                $stmt->execute([$user['id']]);
                $recent_attempts = $stmt->fetchColumn();
                
                if ($recent_attempts >= 3) {
                    $errors[] = "Too many password reset requests. Please try again in 1 hour.";
                } else {
                    // Generate password reset token
                    $reset_token = generateVerificationToken($user['id'], 'password_reset');
                    $reset_url = BASE_URL . "/reset-password.php?token=" . $reset_token;
                    
                    // Send password reset email
                    $email_sent = sendPasswordResetEmail($email, $user['full_name'], $reset_url);
                    
                    if ($email_sent) {
                        $success = true;
                    } else {
                        $errors[] = "Failed to send reset email. Please try again.";
                    }
                }
            } else {
                // Don't reveal whether email exists for security
                $success = true;
            }
        } catch (PDOException $e) {
            $errors[] = "An error occurred. Please try again.";
            error_log("Password reset request error: " . $e->getMessage());
        }
    }
}

function sendPasswordResetEmail($to, $name, $reset_url) {
    $subject = "Reset Your Password - Oxland Credits";
    
    $body = "
    <!DOCTYPE html>
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #1a472a; color: white; padding: 20px; text-align: center; }
            .content { background: #f9f9f9; padding: 20px; }
            .button { background: #1a472a; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
            .footer { text-align: center; margin-top: 20px; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>Oxland Credits</h1>
            </div>
            <div class='content'>
                <h2>Reset Your Password</h2>
                <p>Hello $name,</p>
                <p>We received a request to reset your password for your Oxland Credits account. Click the button below to create a new password:</p>
                <p style='text-align: center;'>
                    <a href='$reset_url' class='button'>Reset Password</a>
                </p>
                <p>Or copy and paste this link in your browser:<br>
                <code>$reset_url</code></p>
                <p>This password reset link will expire in 1 hour.</p>
                <p>If you didn't request a password reset, please ignore this email. Your account remains secure.</p>
            </div>
            <div class='footer'>
                <p>&copy; 2024 Oxland Credits. All rights reserved.</p>
            </div>
        </div>
    </body>
    </html>";
    
    return sendEmail($to, $subject, $body);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password - Oxland Credits</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <div class="max-w-md w-full space-y-8">
        <div class="text-center">
            <span class="text-3xl font-black text-gray-900">OXLAND</span><span class="text-3xl font-black text-green-800">.</span>
        </div>

        <?php if ($success): ?>
            <div class="bg-white p-8 rounded-2xl shadow-lg border border-gray-200 text-center">
                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-check-circle text-green-600 text-2xl"></i>
                </div>
                
                <h2 class="text-2xl font-bold text-gray-900 mb-4">Check Your Email</h2>
                
                <p class="text-gray-600 mb-6">
                    If an account exists with that email, we've sent password reset instructions.
                </p>
                
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 text-left mb-6">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-info-circle text-blue-400"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm text-blue-700">
                                <strong>Didn't receive the email?</strong> Check your spam folder or wait a few minutes and try again.
                            </p>
                        </div>
                    </div>
                </div>
                
                <a href="login.php" 
                   class="w-full inline-flex justify-center items-center px-4 py-3 border border-transparent text-sm font-medium rounded-lg text-white bg-green-800 hover:bg-green-700 transition-colors duration-300">
                    <i class="fas fa-sign-in-alt mr-2"></i>
                    Return to Login
                </a>
            </div>
            
        <?php else: ?>
            <div class="bg-white p-8 rounded-2xl shadow-lg border border-gray-200">
                <h2 class="text-2xl font-bold text-gray-900 mb-6 text-center">Reset Your Password</h2>
                
                <p class="text-gray-600 mb-6 text-center">
                    Enter your email address and we'll send you instructions to reset your password.
                </p>

                <?php if (!empty($errors)): ?>
                    <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <i class="fas fa-exclamation-circle text-red-400"></i>
                            </div>
                            <div class="ml-3">
                                <ul class="list-disc list-inside text-sm text-red-700 space-y-1">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo htmlspecialchars($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <form method="POST" class="space-y-6">
                    <div>
                        <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                        <input id="email" name="email" type="email" required 
                               class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-800 focus:border-transparent"
                               placeholder="Enter your email address">
                    </div>

                    <button type="submit" 
                            class="w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-lg text-white bg-green-800 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-800 transition-colors duration-300">
                        <i class="fas fa-key mr-2"></i>
                        Send Reset Instructions
                    </button>
                </form>

                <div class="mt-6 text-center">
                    <p class="text-sm text-gray-600">
                        Remember your password? 
                        <a href="login.php" class="font-medium text-green-800 hover:text-green-700">Sign in here</a>
                    </p>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>